/***************************************************************************
**
**  This file is part of gphistogram.
**
**  gphistogram is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gphistogram is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-10-27
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <SciFigs.h>

#include "GridFilter.h"

/*
 *  Constructs a GridFilter as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  true to construct a modal dialog.
 */
GridFilter::GridFilter(QWidget *parent, Qt::WindowFlags f)
    : Dialog(parent, f)
{
  TRACE;
  setupUi(this);
  connect(absoluteThresholdFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(relativeThresholdFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(curveRangeFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(curveRangeType, SIGNAL(currentIndexChanged(int)), this, SLOT(enableWidgets()));
  connect(aboveCurveFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(belowCurveFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(insideCurvesFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
  connect(outsideCurvesFilter, SIGNAL(toggled(bool)), this, SLOT(enableWidgets()));
}

void GridFilter::setCurveList(const QList<CurveProperties *>& list)
{
  TRACE;
  for(QList<CurveProperties *>::const_iterator it=list.begin(); it!=list.end(); it++) {
    curve1Name->addItem((*it)->proxy()->name());
    curve2Name->addItem((*it)->proxy()->name());
  }
  if(curve1Name->count()==0) {
    curveRangeFilter->setEnabled(false);
    curveRangeFilter->setChecked(false);
    aboveCurveFilter->setEnabled(false);
    aboveCurveFilter->setChecked(false);
    belowCurveFilter->setEnabled(false);
    belowCurveFilter->setChecked(false);
  }
  if(curve1Name->count()<=1) {
    insideCurvesFilter->setEnabled(false);
    insideCurvesFilter->setChecked(false);
    outsideCurvesFilter->setEnabled(false);
    outsideCurvesFilter->setChecked(false);
  }
}

void GridFilter::enableWidgets()
{
  TRACE;
  bool b;
  b=absoluteThresholdFilter->isChecked() && absoluteThresholdFilter->isEnabled();
  absoluteThresholdValue->setEnabled(b);
  b=relativeThresholdFilter->isChecked() && relativeThresholdFilter->isEnabled();
  relativeThresholdValue->setEnabled(b);

  b=curveRangeFilter->isChecked() && curveRangeFilter->isEnabled();
  bool twoCurves=(insideCurvesFilter->isChecked() && insideCurvesFilter->isEnabled()) ||
                 (outsideCurvesFilter->isChecked() && outsideCurvesFilter->isEnabled());
  bool anyCurve=b ||
                (aboveCurveFilter->isChecked() && aboveCurveFilter->isEnabled()) ||
                (belowCurveFilter->isChecked() && belowCurveFilter->isEnabled()) ||
                twoCurves;
  curve1Name->setEnabled(anyCurve);
  curve2Label->setEnabled(twoCurves);
  curve2Name->setEnabled(twoCurves);
  curveRangeType->setEnabled(b);
  curveRangeFactor->setEnabled(b);
  switch(curveRangeType->currentIndex()) {
  default:
    curveRangeFactor->setSuffix("");
    break;
  case 1:
  case 4:
    curveRangeFactor->setSuffix(" %");
    break;
  case 2:
  case 5:
    curveRangeFactor->setSuffix(" sigma");
    break;
  }
}

void GridFilter::on_curve1Name_activated(int)
{
  TRACE;
  if(curve1Name->currentIndex()==curve2Name->currentIndex()) {
    int i=curve1Name->currentIndex()+1;
    if(i>=curve2Name->count()) {
      i-=2;
    }
    curve2Name->setCurrentIndex(i);
  }
}

void GridFilter::on_curve2Name_activated(int)
{
  TRACE;
  if(curve1Name->currentIndex()==curve2Name->currentIndex()) {
    int i=curve2Name->currentIndex()+1;
    if(i>=curve1Name->count()) {
      i-=2;
    }
    curve1Name->setCurrentIndex(i);
  }
}
