/***************************************************************************
**
**  This file is part of geopsyhv.
**
**  geopsyhv is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsyhv is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-09-09
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "HVToolWidget.h"  
#include "HVStation.h"
#include "HVStationSignals.h"
#include "HVSummary.h"
#include "HVResultSheet.h"
#include "FrequencyWindowRejectionWidget.h"

HVToolWidget::HVToolWidget(QWidget * parent) :
    AbstractHVWidget(parent)
{
  TRACE;
  setWindowIcon(QIcon(":hv-22x22.png"));
  setObjectName("HVToolWidget");

  winParam->addComponent(tr("Vertical"));
  winParam->addComponent(tr("North"));
  winParam->addComponent(tr("East"));

  removeRotateStep();

  QMenu * m=selectBut->menu();
  QAction * a=new QAction(tr("Frequency rejection"), this);
  connect(a, SIGNAL(triggered()), this, SLOT(frequencyWindowRejection()));
  m->addAction(a);

  _tool=new HVTool(this);
}

bool HVToolWidget::setSubPool(SubSignalPool * subPool)
{
  TRACE;
  if(!AbstractHVWidget::setSubPool(subPool)) {
    return false;
  }
  setWindowTitle(tr("H/V toolbox - ")+subPool->name());
  return true;
}

AbstractResultSheet * HVToolWidget::createResultSheet()
{
  TRACE;
  HVResultSheet * w=new HVResultSheet;
  connect(w, SIGNAL(peakChanged()), this, SLOT(updateSummary()));
  return w;
}

AbstractSummary * HVToolWidget::createSummary()
{
  TRACE;
  return new HVSummary;
}

/*!
  Frequency domain window rejection suggested by Cox et al. (2020) in GJI
*/
void HVToolWidget::frequencyWindowRejection()
{
  TRACE;
  QList<AbstractStation *> statList=selectedStations();
  updateParameters();
  HVTool * t=static_cast<HVTool *>(tool());

  FrequencyWindowRejectionWidget * d=new FrequencyWindowRejectionWidget(this);
  Settings::getWidget(d);
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    FrequencyWindowRejection& param=t->parameters()->frequencyWindowRejection();
    param.setMinimumFrequency(d->minimumFrequencyEdit->value());
    param.setMaximumFrequency(d->maximumFrequencyEdit->value());
    param.checkFrequencyRange();
    param.setStddevFactor(d->stddevFactorEdit->value());
    param.setMaximumIterationCount(d->maximumIterationCountEdit->value());

    LayerLocker ll(timeWindowLayer());
    QString log;
    StringStream * s=new StringStream(&log);
    App::beginRedirect(s);
    if(!t->frequencyWindowRejection(&statList)) {
      App::endRedirect(s);
      Message::warning(MSG_ID, tr("Frequency window rejection"), tr("Encountered error(s)\n\n%1").arg(log));
    } else {
      App::endRedirect(s);
      App::log(log);
    }
    ll.unlock();
    windowsChanged();
    setWinParamChanged(false);
  }
  delete d;
}
