/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-01-26
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SheetSelectionWidget.h"
#include "GraphicSheet.h"
#include "GraphicObject.h"
#include "GraphicObjectGroup.h"

namespace SciFigs {

#define SELECTION_WEIGHT 3

  /*!
    \class SheetSelectionWidget SheetSelectionWidget.h
    \brief Brief description of class still missing

    Widget used to draw the selection rectangle on top of everything.
    Selection marks are painted by this widget.
  */

  /*!
    Description of constructor still missing
  */
  SheetSelectionWidget::SheetSelectionWidget(QWidget * parent)
    : QWidget(parent)
  {
    TRACE;
    setAttribute(Qt::WA_TransparentForMouseEvents, true);
  }

  /*!
    Description of destructor still missing
  */
  SheetSelectionWidget::~SheetSelectionWidget()
  {
    TRACE;
  }

  void SheetSelectionWidget::addObject(GraphicObject * obj, GraphicObject::SelectionState s)
  {
    TRACE;
    ASSERT(s!=GraphicObject::None);

    if(s==GraphicObject::Hoover) { // Make sur that there is only one hoovered object
      int n=_objects.count();
      for(int i=0; i<n; i++) {
        GraphicObject * obj=_objects.at(i);
        if(obj->selectionState()==GraphicObject::Hoover) {
          _objects.removeAt(i);
          i--;
          n--;
        }
      }
    }
    _objects.append(obj);
    obj->setSelectionState(s);

    /*GraphicObjectList::const_iterator it;
    qDebug() << "Selected objects:";
    for(it=_objects.begin(); it!=_objects.end(); it++) {
      GraphicObject * obj=*it;
      qDebug() << "  " << obj->objectName();
    }*/
    update();
  }

  void SheetSelectionWidget::removeObject(GraphicObject * obj)
  {
    TRACE;
    _objects.removeOne(obj);
    obj->setSelectionState(GraphicObject::None);
    update();
  }

  void SheetSelectionWidget::clear(PropertyProxy * proxy)
  {
    TRACE;
    GraphicObjectList::const_iterator it;
    for(it=_objects.begin(); it!=_objects.end(); it++) {
      GraphicObject * obj=*it;
      obj->setSelectionState(GraphicObject::None);
      if(proxy) {
        obj->removeProperties(proxy);
      }
    }
    _objects.clear();
    update();
  }

  void SheetSelectionWidget::paintEvent(QPaintEvent *)
  {
    TRACE;
    GraphicSheet * sheet=static_cast<GraphicSheet *>(parent());
    QPainter p;
    p.begin(this);

    GraphicObjectGroup * context=qobject_cast<GraphicObjectGroup *>(sheet->context());
    if(context) {
      p.save();
      QFont f=p.font();
      f.setPointSize(24);
      p.setFont(f);
      p.setPen(QColor(200, 200, 200, 150));
      p.drawText(rect(), Qt::AlignCenter, tr("context '%1'").arg(context->objectName()));
      p.restore();
    }

    if(sheet->isMouseTrackingActive()) {
      p.setPen(Qt::DotLine);
      QPoint topLeft=sheet->mouseTrackingRect().topLeft();
      topLeft=sheet->context()->mapToGlobal(topLeft);
      topLeft=mapFromGlobal(topLeft);
      QPoint bottomRight=sheet->mouseTrackingRect().bottomRight();
      bottomRight=sheet->context()->mapToGlobal(bottomRight);
      bottomRight=mapFromGlobal(bottomRight);
      p.drawRect(QRect(topLeft, bottomRight));
    }

    int n=_objects.count();
    for(int i=0; i<n; i++) {
      GraphicObject * obj=_objects.at(i);
      switch (obj->selectionState()) {
      case GraphicObject::None:  // Unselected object should never be in object list
        _objects.removeAt(i);    // but it may happen with aborted enter/exit events
        i--;
        n--;
        continue;
      case GraphicObject::Hoover:
        p.setPen(QPen(Qt::blue,SELECTION_WEIGHT));
        break;
      case GraphicObject::Selected:
        p.setPen(QPen(Qt::gray,SELECTION_WEIGHT));
        break;
      case GraphicObject::Activated:
        p.setPen(QPen(Qt::black,SELECTION_WEIGHT));
        break;
      }
      QPoint pos=obj->parentWidget()->mapToGlobal(obj->pos());
      pos=mapFromGlobal(pos);
      p.drawRect(pos.x()-SELECTION_WEIGHT,
                 pos.y()-SELECTION_WEIGHT,
                 obj->width()+SELECTION_WEIGHT,
                 obj->height()+SELECTION_WEIGHT);

    }
    p.end();
  }

} // namespace SciFigs

