/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-10-30
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "LineItem.h"
#include "LineLayer.h"
#include "AbstractLine.h"
#include "LayerLocker.h"

namespace SciFigs {

/*!
  \class LineItem LineItem.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

LineItem::LineItem(QObject * parent)
   : QAbstractTableModel(parent)
{
  _layer=0;
  _line=0;
}

void LineItem::setLine(LineLayer * layer, AbstractLine * line)
{
  TRACE;
  beginResetModel();
  _layer=layer;
  _line=line;
  endResetModel();
}

int LineItem::rowCount(const QModelIndex &parent) const
{
  TRACE;
  if(!_line) return 0;
  if( !parent.isValid())
    return _line->count();
  return 0;
}

int LineItem::columnCount(const QModelIndex & ) const
{
  TRACE;
  return 4;
}

QVariant LineItem::data(const QModelIndex &index, int role) const
{
  TRACE;
  if(!_line) return QVariant();
  if( !index.isValid()) return QVariant();
  switch (role) {
  case Qt::DisplayRole: {
      Point p=_line->point(index.row(), _layer->pointOptions());
      switch(index.column()) {
      case 1: return p.x();
      case 2: return p.y();
      case 3: return p.z();
      default: return QVariant();
      }
    }
  case Qt::CheckStateRole:
    if(index.column()==0)
      return _line->isValid(index.row()) ? Qt::Checked : Qt::Unchecked;
    else
      return QVariant();
  case Qt::TextAlignmentRole:
    if(index.column()==0)
      return (int)(Qt::AlignHCenter | Qt::AlignVCenter);
    else
      return (int)(Qt::AlignRight | Qt::AlignVCenter);
  default:
    return QVariant();
  }
}

bool LineItem::setData (const QModelIndex & index, const QVariant & value, int role)
{
  TRACE;
  if(!_line) return false;
  if( !index.isValid()) return false;
  switch (role) {
  case Qt::EditRole: {
      LayerLocker ll(_layer);
      switch(index.column()) {
      case 0: return false;
      case 1: _line->setX(index.row(), value.toDouble()); break;
      case 2: _line->setY(index.row(), value.toDouble(), _layer->pointOptions()); break;
      case 3: _line->setZ(index.row(), value.toDouble(), _layer->pointOptions()); break;
      default: break;
      }
      _layer->deepUpdate();
      emit dataChanged(index, index);
      return true;
    }
  case Qt::CheckStateRole:
    if(index.column()==0) {
      LayerLocker ll(_layer);
      if(value.toInt()==Qt::Checked)
        _line->setValid(index.row(), true);
      else
        _line->setValid(index.row(), false);
     _layer->deepUpdate();
     emit dataChanged(createIndex(index.row(), 0), createIndex(index.row(), 3));
      return true;
    } else
      return false;
  default:
    return false;
  }
}

QVariant LineItem::headerData(int section, Qt::Orientation orientation, int role) const
{
  TRACE;
  if(role!=Qt::DisplayRole) return QVariant();
  if(orientation==Qt::Horizontal) {
    switch (section) {
    case 0: return tr("Valid");
    case 1: return tr("X");
    case 2: return tr("Y");
    case 3: return tr("dY");
    default: return 0;
    }
  } else {
    return section + 1;
  }
}

Qt::ItemFlags LineItem::flags (const QModelIndex & index) const
{
  if(!_line) return 0;
  if(index.column()==0) {
    return Qt::ItemIsSelectable | Qt::ItemIsUserCheckable | Qt::ItemIsEnabled;
  } else {
    if(_line->isValid(index.row())) {
      return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
    } else {
      return Qt::ItemIsSelectable;
    }
  }
}

} // namespace SciFigs
