/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-27
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "IrregularGrid2DDraw.h"
#include "GraphContentOptions.h"

namespace SciFigs {

/*!
  \class IrregularGrid2DDraw IrregularGrid2DDraw.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  \fn IrregularGrid2DDraw::IrregularGrid2DDraw()
  Description of constructor still missing
*/

/*!
  \fn IrregularGrid2DDraw::~IrregularGrid2DDraw()
  Description of destructor still missing
*/

/*!
  Returns the x and y index limits of the visible area.
  The grid index order is the same as the axis, hence there is no need to
  check the correctness of startix&lt;stopix, this is always true.
*/
void IrregularGrid2DDraw::setVisibleArea(const IrregularGrid2D & grid, const GraphContentOptions& gc)
{
  TRACE;
  startix=grid.indexOfX(gc.xVisMin());
  stopix=grid.indexOfX(gc.xVisMax());
  startix--;
  stopix++;
  if(startix<0) startix=0;
  if(stopix>=grid.nx()) stopix=grid.nx()-1;
  ASSERT(stopix-startix+1>0);

  startiy=grid.indexOfY(gc.yVisMin());
  stopiy=grid.indexOfY(gc.yVisMax());
  startiy--;
  stopiy++;
  if(startiy<0) startiy=0;
  if(stopiy>=grid.ny()) stopiy=grid.ny()-1;
  ASSERT(stopiy-startiy+1>0);
}

/*!
  Returns the x and y screen coordinates of the cell nodes inside the visible area
  start/stop/ix/iy must have been initialized before calling this function (see getVisibleArea())
*/
void IrregularGrid2DDraw::setCellNodes(const IrregularGrid2D & grid, const GraphContentOptions& gc)
{
  TRACE;
  // Create two int vectors containing the x and y of nodes (centers of blocks)
  nx=stopix-startix+1;
  ASSERT(nx>0);
  nodex=new int[nx];
  int * ptr=nodex;
  for(int ix=startix; ix<=stopix; ix++, ptr++) {
    *ptr=gc.xr2s(grid.x(ix));
  }

  ny=stopiy-startiy+1;
  ASSERT(ny>0);
  nodey=new int[ny];
  ptr=nodey;
  for(int iy=startiy; iy<=stopiy; iy++, ptr++) {
    *ptr=gc.yr2s(grid.y(iy));
  }
}

/*!
  Returns the x and y limits of the cells inside the visible areaas well as the screen coordinates of the
  cell nodes.
  start/stop/ix/iy must have been initialized before calling this function (see getVisibleArea() and
  getCellNodes())
*/
void IrregularGrid2DDraw::setCellLimits()
{
  TRACE;
  // Calculate the limits of blocks in for x axis (limx)
  limx=new int [ nx + 1 ];
  limx[ 0 ]=(3 * nodex[ 0 ] - nodex[ 1 ] ) >> 1;
  for(int ix=1;ix < nx;ix++ )
    limx[ ix ]=(nodex[ ix - 1 ] + nodex[ ix ] ) >> 1;
  limx[ nx ]=(3 * nodex[ nx - 1 ] - nodex[ nx - 2 ] ) >> 1;

  // Calculate the limits of blocks in for y axis (limy)
  limy=new int [ ny + 1 ];
  limy[ 0 ]=(3 * nodey[ 0 ] - nodey[ 1 ] ) >> 1;
  for(int iy=1;iy < ny;iy++ )
    limy[ iy ]=(nodey[ iy - 1 ] + nodey[ iy ] ) >> 1;
  limy[ ny ]=(3 * nodey[ ny - 1 ] - nodey[ ny - 2 ] ) >> 1;
}

} // namespace SciFigs
