/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-05-07
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "CoordTip.h"
#include "GraphContent.h"
#include "AxisWindow.h"

namespace SciFigs {

CoordTipTip::CoordTipTip() :
    QLabel(nullptr, Qt::ToolTip)
{
  TRACE;
  setObjectName("coordTip");
  setAttribute(Qt::WA_DeleteOnClose);

  setPalette(QToolTip::palette());

  setMouseTracking(true);
  setCursor(QCursor(Qt::CrossCursor));
}

bool CoordTipTip::event(QEvent * e)
{
  TRACE;
  switch (e->type()) {
  case QEvent::MouseMove:
  case QEvent::MouseButtonPress:
  case QEvent::MouseButtonRelease:
  case QEvent::MouseButtonDblClick:
  case QEvent::Enter:
  case QEvent::Leave:
  case QEvent::KeyPress:
  case QEvent::KeyRelease:
  case QEvent::WindowDeactivate:
    close();
    return true;
  default:
    return QWidget::event(e);
  }
}

CoordTip::CoordTip()
    : _coordTipTimer(this),
    _coordTipShowTimer(this)
{
  TRACE;
  _enabled=true;
  _coordTip=nullptr;
  _currentGC=nullptr;
  connect(&_coordTipTimer, SIGNAL(timeout()), this, SLOT(fallAsleep()));
  connect(&_coordTipShowTimer, SIGNAL(timeout()), this, SLOT(hideTip()));
  _coordTipTimer.setSingleShot(true);
  _coordTipShowTimer.setSingleShot(true);
}

CoordTip::~CoordTip()
{
  TRACE;
  delete _coordTip;
}

void CoordTip::setEnabled(bool e)
{
  TRACE;
  _enabled=e;
  if(!_enabled) hideTip();
}

void CoordTip::setMousePos(GraphContent * gc, QPoint& pt)
{
  TRACE;
  if(_mousePos!=pt && _enabled) {
    hideTip();
    _mousePos=pt;
    _currentGC=gc;
    if(_coordTipTimer.isActive()) {
      _coordTipTimer.stop();
    }
    _coordTipTimer.start(1000);
  }
}

void CoordTip::fallAsleep()
{
  TRACE;
  hideTip();
  if(_currentGC && _currentGC->rect().contains(_mousePos) ) {
    _coordTip=new CoordTipTip();
    connect(_coordTip, SIGNAL(destroyed()), this, SLOT(hidden()) );
    _coordTip->setText(_currentGC->coordinateTipText(_mousePos));
    QPoint p=_currentGC->mapToGlobal(_mousePos);
#if(QT_VERSION >= QT_VERSION_CHECK(5, 10, 0))
    QScreen * scr=Application::screenAt(p);
    QSize cts=_coordTip->sizeHint();
    if(scr) {
      QRect r=scr->geometry();
      if(p.x()+10+cts.width()>r.right()) {
        if(p.y()+10+cts.height()>r.bottom()) {
          p-=QPoint(cts.width()+10, cts.height()+10);
        } else {
          p-=QPoint(cts.width()+10, -10);
        }
      } else {
        if(p.y()+10+cts.height()>r.bottom()) {
          p-=QPoint(-10, cts.height()+10);
        } else {
          p+=QPoint(10, 10);
        }
      }
    }
#else
    p+=QPoint(10, 10);
#endif
    _coordTip->move(p);
    _coordTip->show();
    _coordTipShowTimer.start(60000);
  }
}

void CoordTip::hideTip()
{
  TRACE;
  if(_coordTipTimer.isActive()) {
    _coordTipTimer.stop();
  }
  if(_coordTip) {
    _coordTip->deleteLater();
    _coordTip=nullptr;
  }
}

void CoordTip::hidden()
{
  TRACE;
  _coordTip=nullptr;
}

} // namespace SciFigs
