/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-07-04
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ColorMapProperties.h"
#include "XMLSciFigs.h"
#include "ListInterpole.h"

namespace SciFigs {

ColorMapProperties::ColorMapProperties(QWidget * parent)
    : PropertyWidget(parent)
{
  TRACE;
  setupUi(this);
  paletteEditor->setPalette(&_current);
  connect(paletteEditor, SIGNAL(beginPaletteChange()), this, SLOT(beginPaletteChange()));
  connect(paletteEditor, SIGNAL(endPaletteChange()), this, SLOT(endPaletteChange()));

  addProperty(Map, colorMapTable);
}

void ColorMapProperties::beginPaletteChange()
{
  TRACE;
  _current=colorMapTable->colorMap();
}

void ColorMapProperties::endPaletteChange()
{
  TRACE;
  colorMapTable->setColors(_current);
}

PropertyValue::WidgetType ColorMapProperties::determineCustomWidgetType(int pid, QWidget *, QWidget * )
{
  if(pid==Map) {
    return PropertyValue::Custom0;
  } else {
    return PropertyValue::Unknown;
  }
}

QVariant ColorMapProperties::customWidgetValue(PropertyValue& p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    return QVariant::fromValue(colorMapTable->colorMap());
  } else {
    return QVariant();
  }
}

bool ColorMapProperties::connectCustomWidget(PropertyValue& p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    QObject::connect(colorMapTable, SIGNAL(changed()), &p, SLOT(touched()));
    return true;
  } else {
    return false;
  }
}

bool ColorMapProperties::setCustomWidget(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    ColorMap map=p.value().value<ColorMap>();
    colorMapTable->setColors(map);
    colorMapTable->setValues(map);
    colorNumber->setValue(map.count());
    colorMapTable->setConstant(p.constant());
    return true;
  } else {
    return false;
  }
}

void ColorMapProperties::setColorMap(const ColorMap& p)
{
  colorMapTable->setColors(p);
  colorMapTable->setValues(p);
  colorNumber->setValue(p.count());
}

ColorMap ColorMapProperties::colorMap() const
{
  return colorMapTable->colorMap();
}

void ColorMapProperties::on_loadPalette_clicked()
{
  TRACE;
  static QString title=tr("Open an existing palette");
  QString fileName=Message::getOpenFileName(title, tr("Color palette (*.pal)"));
  if(!fileName.isEmpty()) {
    ColorMap pal;
    XMLErrorReport xmler(XMLErrorReport::Read);
    xmler.setTitle(title);
    xmler.setFileName(fileName);
    XMLSciFigs s;
    if(xmler.exec(s.restoreFile(fileName, &pal, XMLSciFigs::Data))) {
      colorMapTable->setColors(pal);
      colorMapTable->setValues(pal);
      colorNumber->setValue(pal.count());
    }
  }
}

void ColorMapProperties::on_savePalette_clicked()
{
  TRACE;
  static QString title=tr("Save palette");
  QString fileName=Message::getSaveFileName(title, tr("Color palette (*.pal)"));
  if(!fileName.isEmpty()) {
    XMLErrorReport xmler(XMLErrorReport::Write);
    xmler.setTitle(title);
    xmler.setFileName(fileName);
    XMLSciFigs s;
    ColorMap pal=colorMapTable->colorMap();
    xmler.exec(s.saveFile(fileName, &pal, XMLSciFigs::Data));
  }
}

void ColorMapProperties::on_linearValues_clicked()
{
  TRACE;
  ColorMap pal=colorMapTable->colorMap();
  ListInterpole * d=new ListInterpole(QApplication::activeWindow());
  d->setValues(pal);
  Settings::getWidget(d);
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    int imin=d->fromIndex();
    int imax=d->toIndex();
    pal.setValues(imin, imax, pal.upperValue(imin), pal.upperValue(imax), SamplingParameters::Linear);
    colorMapTable->setValues(pal);
  }
  delete d;
}

void ColorMapProperties::on_logValues_clicked()
{
  TRACE;
  ColorMap pal=colorMapTable->colorMap();
  ListInterpole * d=new ListInterpole(QApplication::activeWindow());
  d->setValues(pal);
  Settings::getWidget(d);
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    int imin=d->fromIndex();
    int imax=d->toIndex();
    pal.setValues(imin, imax, pal.upperValue(imin), pal.upperValue(imax), SamplingParameters::Log);
    colorMapTable->setValues(pal);
  }
  delete d;
}

void ColorMapProperties::on_colorNumber_valueChanged(int)
{
  TRACE;
  colorMapTable->setColorCount(colorNumber->value());
}

} // namespace SciFigs
