/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-08-23
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef GridSearch_H
#define GridSearch_H

#include "Point2D.h"
#include "FunctionSearch.h"

namespace QGpCoreMath {

class GridFunction2;

class QGPCOREMATH_EXPORT GridSearch : public FunctionSearch
{
public:
  GridSearch();
  ~GridSearch();

  void setGrid(double minX, double maxX, double dX,
               double minY=0.0, double maxY=0.0, double dY=1.0,
               double minZ=0.0, double maxZ=0.0, double dZ=1.0);

  int nx() const {return _nx;}
  int ny() const {return _ny;}
  int nz() const {return _nz;}
  int nxy() const {return _nxy;}
  int nxyz() const {return _nxyz;}

  inline Point coordinates(int index) const;

  double value(const Point& p) const {return FunctionSearch::value(p);}
  double value(const Point& p, int index) const {return FunctionSearch::value(p, index);}
  double value(int index) const {return FunctionSearch::value(coordinates(index), index);}

  virtual void globalMax(double absThres=0.0);
  virtual void localMax(int nMax, double absThres=0.0, double relThres=0.0);
  virtual void localMaxBest(int nMax, double absThres=0.0, double relThres=0.0);
protected:
  int _nx, _ny, _nz, _nxy, _nxyz, _diag, _last;
  double _minX, _minY, _minZ, _dX, _dY, _dZ;
  int _neighbors[26];
private:
  inline void refineMax(int k, double maxVal);
  bool isOnEdge(int k);
  double refineMax(const Point& min, const Point& max, double maxVal);
  inline void checkForMax(const Point& p, double& maxVal);

  inline void setNeighbors1(int k);
  inline void setNeighbors2(int k);
  inline void setNeighbors3(int k);
};

inline Point GridSearch::coordinates(int index) const
{
  int iz=index/_nxy;
  index-=iz*_nxy;
  int iy=index/_nx;
  int ix=index-iy*_nx;
  return Point(_minX+_dX*static_cast<double>(ix),
               _minY+_dY*static_cast<double>(iy),
               _minZ+_dZ*static_cast<double>(iz));
}

} // namespace QGpCoreMath

#endif // GRIDSEARCH_H
