/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-11-05
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Cluster.h"

namespace QGpCoreMath {

  /*!
    \class Cluster Cluster.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  Cluster::Cluster(const TriangularMatrix& proximity)
    : _proximity(proximity)
  {
    TRACE;
    int n=_proximity.count();
    _items.resize(n);
    for(int i=0; i<n; i++) {
      _items[i]=new Item(i);
    }
  }

  /*!
    Description of destructor still missing
  */
  Cluster::~Cluster()
  {
    TRACE;
    if(_items.count()>1) { // If not all items are linked in a tree returned by clustering algorithm
      qDeleteAll(_items);
    }
  }

  /*!
    Example from https://en.wikipedia.org/wiki/Single-linkage_clustering
  */
  TriangularMatrix Cluster::singleLinkageTest()
  {
    TriangularMatrix p(5);
    p.at(0, 1)=17;
    p.at(0, 2)=21;
    p.at(0, 3)=31;
    p.at(0, 4)=23;
    p.at(1, 2)=30;
    p.at(1, 3)=34;
    p.at(1, 4)=21;
    p.at(2, 3)=28;
    p.at(2, 4)=39;
    p.at(3, 4)=43;
    return p;
  }

  void Cluster::test(const TriangularMatrix& proximity)
  {
    Cluster c(proximity);
    Cluster::Node * n=c.singleLinkage();
    XMLHeader hdr(n);
    QTextStream(stdout) << hdr.xml_saveString() << Qt::endl;
  }

  /*!
    Searches for the minimum in the upper diagonal.
  */
  QVector<QPoint> Cluster::minima()
  {
    QVector<QPoint> m;
    double minD=std::numeric_limits<double>::infinity();
    int n=_proximity.count();
    for(int i=0; i<n; i++) {
      for(int j=i+1; j<n; j++) {
        double d=_proximity.at(i, j);
        if(d<=minD) {  // Avoid testing two conditions for the majority of cells
          if(d<minD) {
            minD=d;
            m.clear();
            m.append(QPoint(i, j));
          } else if(!m.isEmpty() && i==m.last().x()) {
            m.append(QPoint(i, j));
          }
        }
      }
    }
    if(!isfinite(minD)) {
      m.clear();
    }
    return m;
  }

  /*!
    https://en.wikipedia.org/wiki/Single-linkage_clustering
    Naive implemetation that runs faster enough for our applications.
  */
  Cluster::Node * Cluster::singleLinkage()
  {
    Node * node;
    while(_proximity.count()>1) {
      QVector<QPoint> list=minima();
      node=new Node;
      if(list.isEmpty()) {
        int n=_items.count();
        for(int i=0; i<n; i++) {
          if(!_items[i]->parent()) {
            _items[i]->setParent(node);
          }
        }
        break;
      } else {
        int row=list.last().x();
        int col=list.last().y();
        if(App::verbosity()>=4) {
          mergeReport(row, col);
        }
        _items.at(row)->setParent(node);
        _items[row]=node;
        _items.at(col)->setParent(node);
        _items.removeAt(col);
        removeProximity(row, col);
        for(int i=list.count()-2; i>=0; i--) { // If multiple items in list, they all have the
                                               // same row and they are sorted by increasing col
          row=list.at(i).x();
          col=list.at(i).y();
          if(App::verbosity()>=4) {
            mergeReport(row, col);
          }
          _items.at(col)->setParent(node);
          _items.removeAt(col);
          removeProximity(row, col);
        }
      }
    }
    _items.clear(); // avoid deletion of items by destructor
                    // All items are attached to node
    return node;
  }

  /*!
    Remove row \a row and column \a col and recompute distances to the new cluster in row.
  */
  void Cluster::removeProximity(int row, int col)
  {
    int j;
    for(j=_proximity.count()-1; j>row; j--) {
      double& v1=_proximity.at(row, j);
      double v2=_proximity.constAt(col, j);
      if(v2<v1 && v2>0.0) {
        v1=v2;
      }
    }
    for(j=row-1; j>=0; j--) {
      double& v1=_proximity.at(j, row);
      double v2=_proximity.constAt(col, j);
      if(v2<v1 && v2>0.0) {
        v1=v2;
      }
    }
    _proximity.removeAt(col);
  }

  void Cluster::mergeReport(int row, int col) const
  {
    App::log(tr("Merging...\n"));
    if(_items.at(col)->isContainer()) {
      App::log(tr("    %2\n").arg(static_cast<Node *>(_items.at(col))->toString(80)));
    } else {
      App::log(tr("    %2\n").arg(static_cast<Item *>(_items.at(col))->index()));
    }
    App::log(tr("  with\n"));
    if(_items.at(row)->isContainer()) {
      App::log(tr("    %2\n").arg(static_cast<Node *>(_items.at(row))->toString(80)));
    } else {
      App::log(tr("    %2\n").arg(static_cast<Item *>(_items.at(row))->index()));
    }
  }

    QVector<int> Cluster::Node::indexList() const
    {
      QVector<int> index;
      for(int i=childrenCount()-1; i>=0; i--) {
        const TreeItem * ti=childAt(i);
        if(ti->isContainer()) {
          const Node * child=static_cast<const Node *>(ti);
          index.append(child->indexList());
        } else {
          index.append(static_cast<const Item *>(ti)->index());
        }
      }
      return index;
    }

    QString Cluster::Node::toString(int countMax) const
    {
      ASSERT(countMax>3);
      QVector<int> index=indexList();
      QString s;
      int n=index.count();
      for(int i=0; i<n; i++) {
        s+=" "+QString::number(index.at(i));
      }
      if(s.count()>countMax) {
        s=s.mid(0, countMax-3);
        s+="...";
      }
      return s;
    }

  void Cluster::Item::xml_attributes(XML_ATTRIBUTES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    static const QString keys[]={"index"};
    attributes.add(keys[0], QString::number(_index));
  }

} // namespace QGpCoreMath

