/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  DinverDCCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-07-27
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ModalCurveTarget.h"

namespace DinverDCCore {

  /*!
    \class ModalCurveTarget ModalCurveTarget.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  const QString ModalCurveTarget::xmlModalCurveTargetTag="ModalCurveTarget";

  /*!
    Description of constructor still missing
  */
  ModalCurveTarget::ModalCurveTarget()
      : Target()
  {
    TRACE;
  }

  /*!
    Description of constructor still missing
  */
  ModalCurveTarget::ModalCurveTarget(const ModalCurveTarget& o)
      : Target(o)
  {
    TRACE;
    _curves=o._curves;
  }

  void ModalCurveTarget::addCurve(const ModalCurve& c)
  {
    TRACE;
    _curves.append(c);
    _curves.last().sort();
  }

  void ModalCurveTarget::setCurves(const QList<ModalCurve>& c)
  {
    TRACE;
    _curves=c;
    for(QList<ModalCurve>::iterator it=_curves.begin(); it!=_curves.end(); it++) {
      it->sort();
    }
  }

  void ModalCurveTarget::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
  {
    TRACE;
    for(QList<ModalCurve>::const_iterator it=_curves.begin(); it!=_curves.end(); ++it) {
      it->xml_save(s, context);
    }
  }

  XMLMember ModalCurveTarget::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(context)
    if(tag=="ModalCurve") {
      _curves.append(ModalCurve());
      return XMLMember(&_curves.last());
    } else return Target::xml_member(tag, attributes, context);
  }

  /*!
    First compute the total range of curves and then resamples them according
    to \a step.

    Return false if min and max frequency cannot be deduced
  */
  bool ModalCurveTarget::resample(double step)
  {
    TRACE;
    double amin=std::numeric_limits<double>::infinity(), amax=-std::numeric_limits<double>::infinity(), min, max;
    for(QList<ModalCurve>::const_iterator it=_curves.begin(); it!=_curves.end(); ++it) {
      const ModalCurve& c=*it;
      min=c.minimumX();
      max=c.maximumX();
      if(min<amin) {
        amin=min;
      }
      if(max>amax) {
        amax=max;
      }
    }
    if(amin==std::numeric_limits<double>::infinity() || amax==-std::numeric_limits<double>::infinity() || amin==amax) {
      return false;
    }
    int n=1+qCeil(::log(amax/amin)/::log(step));
    App::log(tr("Resampling targets from %1 to %2 Hz with %3 samples\n").arg(amin).arg(amax).arg(n));
    for(QList<ModalCurve>::iterator it=_curves.begin(); it!=_curves.end(); ++it) {
      ModalCurve& c=*it;
      c.resample(n, amin, amax, LogScale);
    }
    return true;
  }

} // namespace DinverDCCore
