/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-13
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ACTIVEMODELS_H
#define ACTIVEMODELS_H

#include "DinverCoreDLLExport.h"
#include "ModelSet.h"
#include "ActiveModel.h"

namespace DinverCore {

class DINVERCORE_EXPORT ActiveModels : private IncreaseStorage
{
public:
  ActiveModels(ModelSet * allModels);
  ~ActiveModels();

  inline ActiveModel& add(const SetIndex& s, double parentVolume);
  inline ActiveModel& add(const ActiveModel& m);
  void clear();
  void addNavigatorHits(const QVector<ActiveIndex>& indexes);
  void setNavigatorHits(const QVector<ActiveIndex>& indexes);

  void reserve(int n) {IncreaseStorage::reserve(n);}
  int count() const {return IncreaseStorage::size();}
  int parameterCount() const {return _allModels->parameterCount();}
  int targetCount() const {return _allModels->targetCount();}
  ModelSet * allModels() const {return _allModels;}

  ActiveModel& at(const ActiveIndex& i) {return _infos[i.value()];}
  const ActiveModel& at(const ActiveIndex& i) const {return _infos[i.value()];}

  inline double misfit(const ActiveIndex& i, const TargetIndex& t) const;
  inline const int * model(const ActiveIndex& i) const;

  void print(const ActiveIndex& i);
  void printNavigatorHits() const;
private:
  virtual void reallocate();

  ModelSet * _allModels;
  ActiveModel * _infos;
};

inline ActiveModel& ActiveModels::add(const SetIndex& s, double parentVolume)
{
  _infos[size()].init(s, parentVolume);
  IncreaseStorage::add();
  return _infos[size()-1]; // reallocate possible with add
}

inline ActiveModel& ActiveModels::add(const ActiveModel& m)
{
  _infos[size()]=m;
  IncreaseStorage::add();
  return _infos[size()-1]; // reallocate possible with add
}

inline double ActiveModels::misfit(const ActiveIndex& i, const TargetIndex& t) const
{
  return _allModels->misfit(_infos[i.value()].index(), t);
}

inline const int * ActiveModels::model(const ActiveIndex& i) const
{
  return _allModels->model(_infos[i.value()].index());
}

} // namespace DinverCore

#endif // ACTIVEMODELS_H
