/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-21
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRayleighFixedEll.h"
#include "FKGridSearch.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKSteeringThreeComponentRayleighFixedEll.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRayleighFixedEll HRFKDirectRayleighFixedEll.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  bool HRFKDirectRayleighFixedEll::setFixedEllipticity(double frequency)
  {
    if(_fixedEllipticities.isEmpty()) {
      return true;
    } else {
      if(_fixedEllipticities.isInsideRange(frequency)) {
        setEllipticity(_fixedEllipticities.y(frequency));
        return true;
      } else {
        App::log(tr("  Frequency %1 Hz out of fixed ellipticity range\n").arg(frequency));
      }
    }
    return false;
  }

  void HRFKDirectRayleighFixedEll::setGrid(FKGridSearch * g, double step, double size)
  {
    // Classical kx, ky
    g->setGrid(-size, size, step, -size, size, step);
  }

  bool HRFKDirectRayleighFixedEll::initGrid(int n)
  {
    if(_gridCache->isEmpty(FKCache::ThreeComponentRayleighFixedEll)) {
      App::log(tr("Caching steering vectors for 3-component Rayleigh (fixed ell) ... (%1 values)\n").arg(n) );
      _gridCache->resize(FKCache::ThreeComponentRayleighFixedEll, n);
      _gridCache->reportCacheVolume(FKCache::ThreeComponentRayleighFixedEll, n,
                                    FKSteeringThreeComponentRayleighFixedEll::sizeFactor);
      return true;
    } else {
      return false;
    }
  }

  void HRFKDirectRayleighFixedEll::initGrid(const Point& k, int index)
  {
    FKSteeringThreeComponentRayleighFixedEll::init(_gridCache, index , k);
  }

  void HRFKDirectRayleighFixedEll::setEllipticity(double ell)
  {
    _ellipticity=ell;
    // Build a matrix with factors to dot multiply cross-spectral matrix
    // I did not check in details but this way is maybe faster than directly
    // assigning values to _ellipticityCrossSpectrum due to at() function
    // that require an index multiplication, while dotMultiply() uses simple indexes.
    int n=_crossSpectrum->array().count();
    int n2=2*n;
    int n3=3*n;
    ComplexMatrix ellMat(n3);
    Complex cell(0.0, -ell);
    Complex ccell(0.0, ell);
    Complex cell2(ell*ell, 0.0);
    Complex unit(1.0, 0.0);
    for(int i=0; i<n2; i++) {
      for(int j=0; j<n2; j++) {
        ellMat.at(i, j)=cell2;
      }
    }
    for(int i=n2; i<n3; i++) {
      for(int j=0; j<n2; j++) {
        ellMat.at(i, j)=cell;
        ellMat.at(j, i)=ccell;
      }
      for(int j=n2; j<n3; j++) {
        ellMat.at(i, j)=unit;
      }
    }
    _crossSpectrum->setEllipticity(ellMat);
  }

  double HRFKDirectRayleighFixedEll::value(const Point& k, int index) const
  {
    // General grid is not compatible with the local grids
    // For testing, skip grid cache.
    return value(k);
    if(isInsideLimits(k)) {
      FKPower p(&FKSteeringThreeComponentRayleighFixedEll::cache(_gridCache, index));
      p.setThreeComponentValue(_crossSpectrum->matrix());
      return p.value();
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighFixedEll::value(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKSteeringThreeComponentRayleighFixedEll * s=static_cast<FKSteeringThreeComponentRayleighFixedEll *>(_steering);
      s->initValue(k);
      FKPower p(s);
      p.setThreeComponentValue(_crossSpectrum->matrix());
      return p.value();
    } else {
      return -1.0;
    }
  }


  bool HRFKDirectRayleighFixedEll::loadFixedEllipticityCurve(const QString& fileName)
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      QTextStream s(&f);
      LineParser lp;
      _fixedEllipticities.clear();
      bool ok=true;
      int lineNumber=0;
      while(!s.atEnd()) {
        QString buf=s.readLine();
        lineNumber++;
        if(!buf.isEmpty() && buf[0]!='#') {
          lp.setString(buf);
          Point2D p(lp.toDouble(0, ok), lp.toDouble(1, ok));
          if(!ok) {
            App::log(tr("Error parsing line %1 in file '%2'\n").arg(lineNumber).arg(fileName));
            return false;
          }
          _fixedEllipticities.append(p);
        }
      }
      return true;
    } else {
      App::log(tr("Cannot open file '%1'\n").arg(fileName));
      return false;
    }
  }

} // namespace ArrayCore

