/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-11-29
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKWorker.h"
#include "FKTaskManager.h"
#include "FKGridSearch.h"
#include "ArrayStations.h"
#include "WaveNumberConverter.h"
#include "FKResults.h"

//#define INSPECT_FREQUENCY 2.945
//#define INSPECT_TIME "20170314115338"

namespace ArrayCore {

  /*!
    \class FKWorker FKWorker.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  FKWorker::FKWorker(const ArraySelection * array,
                     const FKParameters * param)
  {
    TRACE;
    _crossSpectrum=new FKCrossSpectrum(array, param);
    _peaks.setTimeReference(array->array()->first()->minTime());
    _grid=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  FKWorker::~FKWorker()
  {
    TRACE;
  }

  bool FKWorker::setGrid(FKCache * gridCache)
  {
    TRACE;
    AbstractFKFunction * f=AbstractFKFunction::create(taskManager()->array().array()->mode(),
                                                      crossSpectrum()->parameters(), gridCache);
    if(f) {
      f->setCrossSpectrum(crossSpectrum());
      const FKParameters * param=crossSpectrum()->parameters();
      // Cross-spectrum depends upon k for RTBF, analytical gradient is not supported.
      /*if(param->inversionMethod()==FKParameters::Gradient && param->processType()!=FKParameters::RTBF) {
         g=new FKGradientSearch;
         g->setFunction(f);
         f->setGrid(static_cast<FKGradientSearch *>(g), param->cacheGridStep(), param->gridStep(), param->effectiveGridSize());
      } else {*/
         _grid=new FKGridSearch;
         _grid->setFunction(f);
         f->setGrid(_grid, param->gridStep(), param->effectiveGridSize());
      //}
      return true;
    } else {
      delete _grid;
      _grid=nullptr;
      return false;
    }
  }

  void FKWorker::process(const QVector<int>& blocks)
  {
    // If the frequency band width is null, _crossSpectrum->frequency().center() is not exactly the
    // frequency that is processed. Using DTFT exact frequency is required.
    double freq;
    if(_crossSpectrum->isSingleValueSpectrum()) {
      freq=_crossSpectrum->timeWindows()->frequencyFilter().exactFrequency();
    } else {
      freq=_crossSpectrum->timeWindows()->frequency().center();
    }
    WaveNumberConverter conv(freq);
    const TimeWindowList& winList=_crossSpectrum->timeWindows()->list();
    conv.setTime(winList.at(blocks.first()).start());

#ifdef INSPECT_FREQUENCY
    if(fabs(conv.frequency()-INSPECT_FREQUENCY)<0.1) {
      App::log(tr("Inspect selected frequency %1 Hz\n").arg(conv.frequency()));
    } else {
      return;
    }
#endif
#ifdef INSPECT_TIME
    DateTime t;
    t.fromString(INSPECT_TIME);
    if(fabs(conv.time().secondsTo(t))<0.1) {
      App::log(tr("Inspect time window starting at %1\n").arg(conv.time().toString()));
    } else {
      return;
    }
#endif
    if(!setSlownessRange()) {
      return;
    }
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_grid->function());
    function->setTotalLimits(crossSpectrum()->timeWindows()->frequency().squaredOmega());
    function->setAzimuthRange(crossSpectrum()->parameters()->minimumAzimuth(),
                              crossSpectrum()->parameters()->maximumAzimuth());
    if(crossSpectrum()->calculate(blocks, function)) {
      localizeMax();
      if(!addPeak(conv)) {
        return;
      }
    }
  }

  void FKWorker::mergeResults()
  {
    FKResults * results=static_cast<FKResults *>(taskManager()->results());
    if(results) {
      results->add(_peaks);
    }
  }

  AbstractFKFunction * FKWorker::function()
  {
    if(setSlownessRange()) {
      AbstractFKFunction * f=static_cast<AbstractFKFunction *>(_grid->function());
      f->setTotalLimits(crossSpectrum()->timeWindows()->frequency().squaredOmega());
      f->setAzimuthRange(crossSpectrum()->parameters()->minimumAzimuth(),
                         crossSpectrum()->parameters()->maximumAzimuth());
      return f;
    } else {
      return nullptr;
    }
  }

  void FKWorker::localizeMax()
  {
    TRACE;
    const FKParameters * param=crossSpectrum()->parameters();
    if(param->maximumPeakCount()>1) {
      AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_grid->function());
      if(function->hasEllipticity()) {
        _grid->localMaxBest(param->maximumPeakCount(),
                            param->absoluteThreshold(),
                            param->relativeThreshold());
      } else {
        _grid->localMax(param->maximumPeakCount(),
                        param->absoluteThreshold(),
                        param->relativeThreshold());
      }
    } else {
      _grid->globalMax(param->absoluteThreshold());
    }
  }

  bool FKWorker::addPeak(const WaveNumberConverter& conv)
  {
    TRACE;
    const FKParameters * param=static_cast<const FKParameters *>(taskManager()->parameters());
    //FKResults * results=static_cast<FKResults *>(_taskManager->results());
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_grid->function());
    FunctionMaximaIterator it;
    bool ok;
    APP_LOG(4, tr("    Found %1 peaks\n").arg(_grid->maximaCount()))
    for(it=_grid->begin(); it!=_grid->end(); ++it) {
      const FunctionSearchMaximum& m=**it;
      Point kell=m.pos();
      APP_LOG(4, tr("    New peak at k=%1\n").arg(kell.toString()))
      if(!function->isOnLimit(kell, conv.frequency())) {
        double ell=function->ellipticity(kell, ok); // Ensure that ellipticity is computed before
                                                    // everything else: noise,...
        if(ok || param->saveUndefinedEllipticities()) {
          if(!function->refine(kell, ell, &_peaks, conv)) {
            if(!_peaks.add(function->waveNumber(kell),
                           ell,
                           function->noise(kell),
                           function->power(m.value()),
                           conv,
                           param->minimumSlowness())) {
              return false;
            }
          }
        }
      }
    }
    return true;
  }

  bool FKWorker::setSlownessRange()
  {
    TRACE;
    const FKParameters * param=crossSpectrum()->parameters();
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_grid->function());
    double f=crossSpectrum()->timeWindows()->frequency().center();
    if(param->isRefine()) {
      //App::log(tr("Looking for frequency %1 in refinement limits\n").arg(f)
      //              << param->refineMinimumSlowness().toString() << endl;
      int index=param->refineMinimumSlowness().indexOf(f, 1e-4); // default frequency step is 2.5%
      if(index<0) {
        APP_LOG(1, tr("  Skipped frequency %1 Hz\n").arg(f))
        return false;
      } else {
        function->setMinimumSlowness(param->refineMinimumSlowness().y(index));
        function->setMaximumSlowness(param->refineMaximumSlowness().y(index));
        APP_LOG(1, tr("  Minimum slowness %1 s/m at %2 Hz\n").arg(function->minimumSlowness()).arg(f))
        APP_LOG(1, tr("  Maximum slowness %1 s/m at %2 Hz\n").arg(function->maximumSlowness()).arg(f))
        return true;
      }
    } else {
      // Rayleigh and Love must be computed on the same frequency range but
      // results may accept values on distinct frequency ranges.
      if(param->acceptFrequency(f)) {
        return true;
      } else {
        APP_LOG(1, tr("  Skipped frequency %1 Hz\n").arg(f))
        return false;
      }
    }
  }

} // namespace ArrayCore

