/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-11-22
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKCache.h"
#include "FKSteeringOneComponent.h"
#include "FKSteeringTwoComponentRayleighRadial.h"
#include "FKSteeringThreeComponentRayleighRadial.h"
#include "FKSteeringThreeComponentRayleighFixedEll.h"
#include "FKSteeringTwoComponentRadial.h"
#include "FKSteeringTwoComponentTransverse.h"
#include "FKSteeringTwoComponentRayleighFixedEll.h"

namespace ArrayCore {

  /*!
    \class FKCache FKCache.h
    \brief Steering matrix for various FK methods

    Centralized storage for the steering vectors.

  */

  /*!
    Description of constructor still missing
  */
  FKCache::FKCache(const ArraySelection * array)
  {
    TRACE;
    _array=array;
    for(int i=0; i<STEERING_CASE_COUNT; i++) {
      _steering[i]=nullptr;
    }
  }

  /*!
    Description of destructor still missing
  */
  FKCache::~FKCache()
  {
    TRACE;
    for(int i=0; i<STEERING_CASE_COUNT; i++) {
      delete [] _steering[i];
    }
  }

  /*!

  */
  void FKCache::resize(SteeringType st, int cacheItemCount)
  {
    FKSteering *& s=_steering[st];
    ASSERT(!s);
    switch(st) {
    case OneComponent:
      s=new FKSteeringOneComponent[static_cast<size_t>(cacheItemCount)];
      break;
    case TwoComponentRayleigh:
      s=new FKSteeringTwoComponentRayleighRadial[static_cast<size_t>(cacheItemCount)];
      break;
    case ThreeComponentRayleigh:
      s=new FKSteeringThreeComponentRayleighRadial[static_cast<size_t>(cacheItemCount)];
      break;
    case ThreeComponentRayleighFixedEll:
      s=new FKSteeringThreeComponentRayleighFixedEll[static_cast<size_t>(cacheItemCount)];
      break;
    case TwoComponentRadial:
      s=new FKSteeringTwoComponentRadial[static_cast<size_t>(cacheItemCount)];
      break;
    case TwoComponentTransverse:
      s=new FKSteeringTwoComponentTransverse[static_cast<size_t>(cacheItemCount)];
      break;
    case TwoComponentRayleighFixedEll:
      s=new FKSteeringTwoComponentRayleighFixedEll[static_cast<size_t>(cacheItemCount)];
      break;
    }
    for(int i=0; i<cacheItemCount; i++) {
      s[i].setArray(_array);
    }
  }

  void FKCache::reportCacheVolume(SteeringType st, int count, int sizeFactor) const
  {
    App::log(tr("  cache volume: %1 Gb\n").arg(steering(st, 0).dataSizeMb(_array->count()*sizeFactor)*count/1024.0));
  }

} // namespace ArrayCore

