/***************************************************************************
**
**  This file is part of spac2disp.
**
**  spac2disp is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  spac2disp is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-06-03
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverCore.h>
#include <DinverDCCore.h>
#include <QGpCoreWave.h>
#include <QGpGuiTools.h>

#include "spac2dispVersion.h"
#include "spac2dispInstallPath.h"
#include "SpacSelector.h"
#include "Spac3CForward.h"
#include "Reader.h"

PACKAGE_INFO("spac2disp", SPAC2DISP)

ApplicationHelp * help();

// Options
static QString pickCurve, outputTarget;
static QString curveParser;
static QStringList loadFileNames;
static double kmin=-1, kmax=-1, vmin=-1, vmax=-1;
static int maxSolutionCount=5, cellCount=-1;
static ExportOptions exportOptions;
static AutocorrCurves autocorr;
static QString dispersionTargetFileName, autocorrTargetFileName;
Reader maxReader;

bool saveDispersions(ModalCurve& rayleigh, ModalCurve& love)
{
  TargetList tl;
  rayleigh.addMode(Mode(Mode::Phase, Mode::Rayleigh, 0));
  love.addMode(Mode(Mode::Phase, Mode::Love, 0));
  tl.dispersionTarget().addCurve(rayleigh);
  tl.dispersionTarget().addCurve(love);
  XMLVirtualPlugin plugin(&tl, "DispersionCurve");
  XMLDinverHeader hdr(&plugin);
  if(hdr.xml_saveFile(dispersionTargetFileName)!=XMLClass::NoError) {
    App::log(tr("spac2disp: error writing to file %1\n").arg(dispersionTargetFileName));
    return false;
  } else {
    return true;
  }
}

bool saveForwardAutocorr(ModalCurve * targets, int curveCount)
{
  for(int ci=0; ci<curveCount;ci++) {
    targets[ci].addMode(autocorr.curves().at(ci).modes().first());
  }
  for(int ci=0; ci<curveCount;ci++) {
    autocorr.curves().append(targets[ci]);
  }
  TargetList autocorrTarget;
  autocorrTarget.autocorrTarget().setCurves(autocorr);
  XMLVirtualPlugin plugin(&autocorrTarget, "DispersionCurve");
  XMLDinverHeader hdr(&plugin);
  if(hdr.xml_saveFile(autocorrTargetFileName)!=XMLClass::NoError) {
    App::log(tr("spac2disp: error writing to file %1\n").arg(autocorrTargetFileName));
    return false;
  } else {
    return true;
  }
}

void setVRTOutputFileName(const QFileInfo& fi)
{
  if(dispersionTargetFileName.isEmpty()) {
    QDir d(fi.absoluteDir());
    dispersionTargetFileName=d.absoluteFilePath(fi.baseName()+"-disp.target");
    autocorrTargetFileName=d.absoluteFilePath(fi.baseName()+"-forward.target");
  }
}

bool loadAutocorr(int& argc, char ** argv, QString& title)
{
  int i, j=1;
  for(i=1; i<argc; i++) {
    QFileInfo fi(argv[i]);
    if(fi.suffix()=="target" || fi.suffix()=="dinver") {
      title+=fi.fileName()+" ";
      TargetList tl;
      XMLVirtualPlugin plugin(&tl, "DispersionCurve");
      XMLDinverHeader hdr(&plugin);
      if(hdr.xml_restoreFile(argv[i])!=XMLClass::NoError) {
        Message::warning(MSG_ID, tr("Loading Dinver target/environment ..."),
                                 tr("Error while reading file %1").arg(argv[i]), Message::cancel());
        return false;
      }
      setVRTOutputFileName(fi);
      autocorr.add(tl.autocorrTarget().curves());
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }

  if(autocorr.isEmpty()) {
    QStringList files=Message::getOpenFileNames(tr("Loading SPAC files"),
                                                tr("Dinver targets (*.target);;Dinver environment (*.dinver)"));
    if(files.isEmpty()) {
      Message::critical(MSG_ID, tr("Loading SPAC files"),
                                tr("No SPAC files provided"));
      return false;
    }
    for(QStringList::iterator it=files.begin(); it!=files.end(); it++) {
      QFileInfo fi(*it);
      title+=fi.fileName()+" ";
      TargetList tl;
      XMLVirtualPlugin plugin(&tl, "DispersionCurve");
      XMLDinverHeader hdr(&plugin);
      if(hdr.xml_restoreFile(*it)!=XMLClass::NoError) {
        Message::warning(MSG_ID, tr("Loading Dinver target/project ..."),
                                 tr("Error while reading file %1").arg(*it), Message::cancel());
        return false;
      }
      setVRTOutputFileName(fi);
      autocorr.add(tl.autocorrTarget().curves());
    }
  }
  return true;
}

int verticalDispersion()
{
  QTextStream sOut(stdout);
  int n=autocorr.curves().count();
  for(int i=0; i<n; i++) {
    ModalCurve c=autocorr.dispersionCurve(i, kmin, kmax, maxSolutionCount);
    const ModalCurve& curve=autocorr.curves().at(i);
    int iRing=curve.modes().first().ringIndex();
    const AutocorrRing& ring=autocorr.ring(iRing);
    sOut << QString("# Curve %1, ring %2 (from %3 to %4)\n")
            .arg(i).arg(iRing).arg(ring.minRadius()).arg(ring.maxRadius())
         << c.toString();
  }
  return 0;
}

int verticalGui(const QString& title)
{
  SciFigsGlobal s;
  SpacSelector * w=new SpacSelector;
  w->setMaximumSolutionCount(maxSolutionCount);
  w->setWavenumberRange(kmin, kmax);
  w->setVelocityRange(vmin, vmax);
  w->setCellCount(cellCount);
  w->setWindowTitle(title);
  if(!w->createObjects(&autocorr)) {
    delete w;
    return 2;
  }
  if(!loadFileNames.isEmpty()) {
    for(QStringList::iterator it=loadFileNames.begin(); it!=loadFileNames.end(); it++) {
      w->loadCurves(*it, curveParser);
    }
  }
  w->exportPlot(exportOptions);
  int appReturn;
  if(exportOptions.exportFile().isEmpty()) {
    w->show();
    appReturn=Application::instance()->exec();
  } else {
    appReturn=0;
  }
  delete w;
  return appReturn;
}

int verticalPickCurve()
{
  SciFigsGlobal s;
  SpacSelector * w=new SpacSelector;
  w->setMaximumSolutionCount(maxSolutionCount);
  w->setWavenumberRange(kmin, kmax);
  w->setVelocityRange(vmin, vmax);
  w->setCellCount(cellCount);
  if(!w->createObjects(&autocorr)) {
    delete w;
    return 2;
  }
  if(!pickCurve.isEmpty()) {
    w->loadCurves(pickCurve, curveParser);
    w->adjustCurve(w->curves->curves().last());
  } else {
    w->autoPick();
  }
  QTextStream sOut(stdout);
  w->curves->curves().last()->proxy()->save(sOut);
  delete w;
  return 0;
}

int allDispersion()
{
  Spac3CForward forward;
  forward.setCurves(autocorr);
  forward.parameterSpace().setVariableParameters();

  int omegaCount=forward.omegasCount();
  int curveCount=autocorr.curves().count();
  ModalCurve rayleigh, love;
  Curve<Point2D> alpha;
  ModalCurve * targets=new ModalCurve[curveCount];
  for(int i=0; i<omegaCount; i++) {
    forward.setFrequency(i);
    static const QString baseName="frequency_%1";
    double f=forward.omega(i)*0.5/M_PI;
    App::freeze(true);
    ModelRepository na;
    na.setForward(&forward);
    na.setStorage();
    na.setMaximumModelCount(50);
    na.openReport(baseName.arg(f)+".report");
    na.start(1, Generator::MonteCarlo);
    na.wait();
    na.setMaximumModelCount(1000);
    na.start(1, Generator::Neighborhood);
    na.wait();
    App::freeze(false);
    if(na.validModelCount()>0) {
      SetIndex iBest=na.bestModelIndex();
      double model[3];
      model[0]=na.variableParameterValue(iBest, 0);
      model[1]=na.variableParameterValue(iBest, 1);
      model[2]=na.variableParameterValue(iBest, 2);
      rayleigh.append(FactoryPoint(f,model[0]));
      love.append(FactoryPoint(f,model[1]));
      alpha.append(Point2D(f,model[2]));
      App::log(tr("# %1-%2 Hz: %3\n").arg(i).arg(f).arg(na.misfit(iBest)));
      forward.setModel(model);
      for(int ci=0; ci<curveCount;ci++) {
        Mode m=autocorr.curves().at(ci).modes().first();
        switch(m.polarization()) {
        case Mode::Vertical:
          targets[ci].append(FactoryPoint(f,forward.vertical(m.ringIndex())));
          break;
        case Mode::Radial:
          targets[ci].append(FactoryPoint(f,forward.radial(m.ringIndex())));
          break;
        case Mode::Transverse:
          targets[ci].append(FactoryPoint(f,forward.transverse(m.ringIndex())));
          break;
        default:
          break;
        }
      }
    }
  }

  QTextStream(stdout) << "# Alpha\n" <<  alpha.toString();
  if(!saveDispersions(rayleigh, love) ||
     !saveForwardAutocorr(targets, curveCount)) {
    delete [] targets;
    return 2;
  } else {
    delete [] targets;
    return 0;
  }
}

int maxSpac2Disp(int argc, char ** argv)
{
  if(vmin>0.0) {
    maxReader.setMinimumVelocity(vmin);
  }
  if(vmax>0.0) {
    maxReader.setMaximumVelocity(vmax);
  }
  maxReader.setSamples(new Samples);
  if(!maxReader.read(argc, argv, maxReader.ignoreStdin())) {
    return 2;
  }
  if(maxReader.isEmpty()) {
    if(!maxReader.read(Message::getOpenFileNames(tr("Open data file"),
                                                 tr("SPAC Max files (*.max)")))) {
      return 2;
    }
  }
  if(maxReader.strictFrequencyRange()) {
    if(!maxReader.setFrequencyRange(autocorr)){
      return 2;
    }
  }
  return maxReader.convert();
}

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  // Options
  enum Spac2dispMode {Gui, Dispersion, PickCurve, SelectTarget, Max};
  enum Component {Vertical, AllComponent};
  Component component=Vertical;
  Spac2dispMode spac2dispMode=Gui;

  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-V") {
        component=Vertical;
      } else if(arg=="-VRT") {
        component=AllComponent;
        spac2dispMode=Dispersion;
      } else if(arg=="-dispersion") {
        spac2dispMode=Dispersion;
      } else if(arg=="-pick") {
        if(CoreApplication::checkOptionArg(i, argc, argv, false)) {
          pickCurve=argv[i];
        }
        spac2dispMode=PickCurve;
      } else if(arg=="-select") {
        CoreApplication::checkOptionArg(i, argc, argv);
        outputTarget=argv[i];
        spac2dispMode=SelectTarget;
      } else if(arg=="-max") {
        component=Vertical;
        spac2dispMode=Max;
      } else if(arg=="-curve-parser") {
        CoreApplication::checkOptionArg(i, argc, argv);
        curveParser=argv[i];
      } else if(arg=="-load-curve") {
        CoreApplication::checkOptionArg(i, argc, argv);
        loadFileNames.append(argv[i]);
      } else if(arg=="-kmin") {
        CoreApplication::checkOptionArg(i, argc, argv);
        kmin=atof(argv[i] );
      } else if(arg=="-kmax") {
        CoreApplication::checkOptionArg(i, argc, argv);
        kmax=atof(argv[i] );
      } else if(arg=="-vmin") {
        CoreApplication::checkOptionArg(i, argc, argv);
        vmin=atof(argv[i] );
      } else if(arg=="-vmax") {
        CoreApplication::checkOptionArg(i, argc, argv);
        vmax=atof(argv[i] );
      } else if(arg=="-cell-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        cellCount=atoi(argv[i] );
      } else if(arg=="-max-solutions") {
        CoreApplication::checkOptionArg(i, argc, argv);
        maxSolutionCount=atoi(argv[i] );
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(!exportOptions.read(argc, argv)) {
    return 2;
  }
  switch(spac2dispMode) {
  case Gui:
  case Dispersion:
  case PickCurve:
  case SelectTarget:
    break;
  case Max:
    if(!maxReader.setOptions(argc, argv)) {
      return 2;
    }
    break;
  }
  if(!CoreApplication::checkRemainingArgs(argc, argv)) {
    return 2;
  }

  // get autocorr curves
  if(spac2dispMode!=Gui) {
    a.setConsoleMessage();
  }
  QString title("spac2disp - ");

  if(!loadAutocorr(argc, argv, title)) {
    return 2;
  }

  int appReturn=0;
  switch(component) {
  case Vertical:
    switch(spac2dispMode) {
    case Gui:
      appReturn=verticalGui(title);
      break;
    case Dispersion:
      appReturn=verticalDispersion();
      break;
    case PickCurve:
      appReturn=verticalPickCurve();
      break;
    case SelectTarget:
      App::log(tr("Not yet implemented\n"));
      break;
    case Max:
      appReturn=maxSpac2Disp(argc, argv);
      break;
    }
    break;
  case AllComponent:
    switch(spac2dispMode) {
    case Gui:
      App::log(tr("Not yet implemented\n"));
      break;
    case Dispersion:
      allDispersion();
      break;
    case PickCurve:
      App::log(tr("Not yet implemented\n"));
      break;
    case SelectTarget:
      App::log(tr("Not yet implemented\n"));
      break;
    case Max:
      App::log(tr("Not yet implemented\n"));
      break;
    }
    break;
  }

  return appReturn;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] <SPAC FILE 1> <SPAC FILE 2> ..." );
  h->setComments( "Validate autocorrelation curves by trying to delineate a common dispersion "
                  "curve (vertical and horizontal components). SPAC FILE can be a .target, "
                  "a .dinver, or a .max file." );
  h->addGroup("Spac2disp","spac2disp");
  h->addOption("-V","Vertical autocorrelation validation (default)");
  h->addOption("-VRT","Vertical, Radial and Tranverse component inversion, "
                      "output '...-dist.target', '...-forward.target' and alpha through stdout.");
  h->addOption("-dispersion","Output the dispersion curve for each input autocorrelation curve. "
                             "Only for vertical component.");
  h->addOption("-pick [<FILE>]","Automatically pick or adjust curve loaded from FILE. Curve is sent to stdout."
                                "Only for vertical component.");
  h->addOption("-select <TARGET>","Selects samples and save the result in file TARGET. Exactly two limiting curves must be "
                                  "specified with option '-load-curve'.");
  h->addOption("-max","Reads a SPAC .max and converts it to a dispersion .max file.");
  h->addOption("-load-curve <FILE>", "Loads curves from FILE. The curve are imported with the parser specified with option '-curve-parser'.");
  h->addOption("-curve-parser <FILE>", "Defines the parser for importing curves");
  h->addOption("-kmin <MIN>","Search for solutions above wave number MIN.");
  h->addOption("-kmax <MAX>","Search for solutions below wave number MAX.");
  h->addOption("-vmin <MIN>","Minimum velocity of dispersion grid.");
  h->addOption("-vmax <MAX>","Maximum velocity of dispersion grid.");
  h->addOption("-cell-count <COUNT>","Number of in dispersion grid.");
  h->addOption("-max-solutions <MAX>","Maximum number of accepted solutions, limits the number of "
                                      "archs in the Bessel function.");
  Reader::help(h);
  return h;
}
