/***************************************************************************
**
**  This file is part of phaseit.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-08-09
**  Authors:
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ToolWidget.h"
#include "Parameters.h"
#include "GridResults.h"
#include "CurveResults.h"

/*!
  \class ToolWidget ToolWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

#define _gridResults static_cast<GridResults*>(_childrenList[0])
#define _curveResults static_cast<CurveResults*>(_childrenList[1])

/*!
  Description of constructor still missing
*/
ToolWidget::ToolWidget(QWidget * parent)
  : AbstractToolWidget(parent, 2)
{
  TRACE;
  setupUi(this);
  _tool=new Tool(this);
  connect(_tool, SIGNAL(finished()), this, SLOT(finish()));

  _trackingRect=nullptr;
  setWindowIcon(QIcon(":phaseit-22x22.png"));

  connect(startBut, SIGNAL(clicked()), this, SLOT(start()));
  connect(stopBut, SIGNAL(clicked()), this, SLOT(stop()));
  connect(loadParam, SIGNAL(clicked()), this, SLOT(loadLogParameters()));

  winParam->addComponent(tr("Any component"));
  winParam->removeFilterOption();
  winParam->setLength(WindowingParameters::Exactly, 60.0);
  winParam->createSelectMenu(selectBut);

  connect(winParam, SIGNAL(autoWindows()), this, SLOT(autoWindows()));
  connect(winParam, SIGNAL(beginAddWindowsManually()), this, SLOT(beginAddWindowsManually()));
  connect(winParam, SIGNAL(endAddWindowsManually()), this, SLOT(endAddWindowsManually()));
  connect(winParam, SIGNAL(beginRemoveWindowsManually()), this, SLOT(beginRemoveWindowsManually()));
  connect(winParam, SIGNAL(endRemoveWindowsManually()), this, SLOT(endRemoveWindowsManually()));
  connect(winParam, SIGNAL(inverseWindows()), this, SLOT(inverseWindows()));
  connect(winParam, SIGNAL(clearWindows()), this, SLOT(clearWindows()));
  connect(winParam, SIGNAL(loadWindows()), this, SLOT(loadWindows()));

  freqSamp->setFrequency();
  frequencyBandwidth->setMinimum(1e-99);
}

void ToolWidget::restoreFields()
{
  TRACE;
  AbstractToolWidget::restoreFields();
  if(timeWindowLayer()) {
    connect(timeWindowLayer(), SIGNAL(addWindows(const Signal *, const TimeRange&)),
            this, SLOT(manualAddWindows(const Signal *, const TimeRange&)));
    connect(timeWindowLayer(), SIGNAL(removeWindows(const Signal *, const TimeRange&)),
            this, SLOT(manualRemoveWindows(const Signal *, const TimeRange&)));
  }
}

void ToolWidget::updateAllFields()
{
  TRACE;
  winParam->updateAllFields();
  freqSamp->updateAllFields();
  timeLimits->setSubPool(subPool());
  timeLimits->updateAllFields();
}

bool ToolWidget::setSubPool(SubSignalPool * subPool)
{
  TRACE;

  QString str=tr("PhaseIt toolbox - ")+subPool->name();
  setWindowTitle(str);

  if(!AbstractToolWidget::setSubPool(subPool)) {
    return false;
  }
  // Set station names in apply list
  for(int i=0; i<tool()->componentCount(); i++) {
    winParam->addComponent(Signal::userName(tool()->component(i)));
  }
  int nStat=tool()->stationCount();
  for(int i=0; i<nStat; i++) {
    winParam->addStation(tool()->station(i)->name());
  }
  setReferenceList();
  for(int i=0; i<nStat; i++) {
    const StationSignals * stat=tool()->station(i);
    if(timeWindowLayer()) {
      timeWindowLayer()->addTimeWindows(stat->originals(0), &tool()->timeWindows());
    }
    winParam->addStation(stat->name());
  }

  _childrenList[0]=new GridResults;
  _gridResults->setObjectName("PhaseItResults");
  _gridResults->setResultCount(nStat-1);
  GeopsyGuiEngine::instance()->addSubWindow(this, _gridResults)->setUserClosable(false);
  _childrenList[1]=new CurveResults;
  _curveResults->setObjectName("PhaseItCurveResults");
  _curveResults->setResultCount(nStat-1);
  GeopsyGuiEngine::instance()->addSubWindow(this, _curveResults)->setUserClosable(false);
  return true;
}


void ToolWidget::setReferenceList()
{
  TRACE;
  stationList->clear();
  int nStat=tool()->stationCount();
  for(int i=0; i<nStat; i++)
    stationList->addItem(tr("%1. %2").arg(i+1).arg(tool()->station(i)->name()));
}

void ToolWidget::windowsChanged()
{
  TRACE;
  if(timeWindowLayer()) timeWindowLayer()->deepUpdate();
  QFont f(startBut->font());
  f.setBold(true);
  startBut->setFont(f);
}

void ToolWidget::beginAddWindowsManually()
{
  TRACE;
  if(timeWindowLayer()) {
    timeWindowLayer()->toggleTrackingAction(true, TimeWindowLayer::Add);
  }
}

void ToolWidget::endAddWindowsManually()
{
  TRACE;
  if(timeWindowLayer()) {
    timeWindowLayer()->toggleTrackingAction(false, TimeWindowLayer::Add);
  }
}

void ToolWidget::beginRemoveWindowsManually()
{
  TRACE;
  if(timeWindowLayer()) {
    timeWindowLayer()->toggleTrackingAction(true, TimeWindowLayer::Remove);
  }
}

void ToolWidget::endRemoveWindowsManually()
{
  TRACE;
  if(timeWindowLayer()) {
    timeWindowLayer()->toggleTrackingAction(false, TimeWindowLayer::Remove);
  }
}

void ToolWidget::manualAddWindows(const Signal *, const TimeRange& r)
{
  TRACE;
  updateParameters();
  QString log=tr("Add windows in range [%1, %2]\n")
                  .arg(r.start().toString(DateTime::defaultUserFormat))
                  .arg(r.end().toString(DateTime::defaultUserFormat));
  LayerLocker ll(timeWindowLayer());
  tool()->addWindows(r);
  ll.unlock();
  windowsChanged();
}

void ToolWidget::manualRemoveWindows(const Signal *, const TimeRange& r)
{
  TRACE;
  updateParameters();
  LayerLocker ll(timeWindowLayer());
  tool()->removeWindows(r);
  ll.unlock();
  windowsChanged();
}

void ToolWidget::autoWindows()
{
  TRACE;
  updateParameters();
  LayerLocker ll(timeWindowLayer());
  tool()->autoWindows();
  ll.unlock();
  windowsChanged();
  selectBut->setText(tr("Select"));
}

void ToolWidget::inverseWindows()
{
  TRACE;
  updateParameters();
  LayerLocker ll(timeWindowLayer());
  tool()->inverseWindows();
  ll.unlock();
  windowsChanged();
}

void ToolWidget::addWindows(const TimeRange& r)
{
  TRACE;
  updateParameters();
  LayerLocker ll(timeWindowLayer());
  tool()->addWindows(r);
  ll.unlock();
  windowsChanged();
}

void ToolWidget::removeWindows(const TimeRange& r)
{
  TRACE;
  LayerLocker ll(timeWindowLayer());
  tool()->removeWindows(r);
  ll.unlock();
  windowsChanged();
}

void ToolWidget::clearWindows()
{
  TRACE;
  LayerLocker ll(timeWindowLayer());
  tool()->clearWindows();
  ll.unlock();
  windowsChanged();
}

void ToolWidget::loadWindows(QString fileName)
{
  TRACE;
  LayerLocker ll(timeWindowLayer());
  tool()->loadWindows(fileName);
  ll.unlock();
  windowsChanged();
}

AbstractParameters * ToolWidget::parameters(AbstractParameters * param) const
{
  TRACE;
  if(!param) {
    param=new Parameters;
  }
  getParameters(*static_cast<Parameters *>(param));
  return param;
}

void ToolWidget::setParameters(const AbstractParameters * param)
{
  TRACE;
  setParameters(*static_cast<const Parameters *>(param));
}

bool ToolWidget::updateParameters()
{
  Parameters param;
  getParameters(param);
  return tool()->setParameters(param);
}

void ToolWidget::getParameters(Parameters& param) const
{
  TRACE;
  timeLimits->getParameters(param.timeRange());
  winParam->getParameters(param.windowing());
  freqSamp->getParameters(param.frequencySampling());
  param.setFrequencyBandwidth(frequencyBandwidth->value());
  if(timeDomain->isChecked()) {
    param.setDomain(Parameters::Time);
  } else {
    param.setDomain(Parameters::Frequency);
  }
  param.setMaximumDelay(maximumDelay->value());
  param.setReferenceIndex(stationList->currentIndex());
}

void ToolWidget::setParameters(const Parameters& param)
{
  TRACE;
  timeLimits->setParameters(param.timeRange());
  winParam->setParameters(param.windowing());
  freqSamp->setParameters(param.frequencySampling());
  timeDomain->setChecked(param.domain()==Parameters::Time);
  frequencyBandwidth->setValue(param.frequencyBandwidth());
  maximumDelay->setValue(param.maximumDelay());
  stationList->setCurrentIndex(param.referenceIndex());
}

bool ToolWidget::hasWindows()
{
  TRACE;
  if (tool()->timeWindows().isEmpty()) {
    if (Message::warning(MSG_ID, windowTitle(),
                         tr("No windows have been selected. Do you want to "
                            "select them automatically with current parameters?"),
                         Message::yes(), Message::no())==Message::Answer0) {
      autoWindows();
    } else return false;
  }
  return true;
}

void ToolWidget::start()
{
  TRACE;
  if(subPoolLocked()) {
    return;
  }
  if(!hasWindows()) return;

  if(!updateParameters()) {
    Message::warning(MSG_ID, tr("Starting"), tr("Error(s) found in parameters, check log."));
    return;
  }
  lockSubPool();
  detailedStatus->setLoop(tool()->loop());
  setRunning(true);
  tool()->start();
}

void ToolWidget::stop()
{
  TRACE;
  tool()->stop();
}

/*!
  To avoid a blank plot for reference (compared with itself), the
  correspondance of stations and result plots changes according
  the index of reference station.
*/
void ToolWidget::setStationPlots()
{
  int nRes=tool()->stationCount()-1;
  int iRef=tool()->parameters()->referenceIndex();
  static const QString templateName(tr("%1 vs %2"));
  for(int iRes=0; iRes<nRes; iRes++) {
    int iStat=iRes<iRef ? iRes : iRes+1;
    _gridResults->setTimeReference(iRes, tool()->station(iStat)->startTime());
    QString name=templateName
        .arg(tool()->station(iStat)->name())
        .arg(tool()->station(iRef)->name());
    _gridResults->setName(iRes, name);
    _curveResults->setName(iRes, name);
  }
}

void ToolWidget::finish()
{
  unlockSubPool();
  setRunning(false);
  setStationPlots();
  _gridResults->setValues(tool()->loop());
  _curveResults->setValues(tool()->loop());
  if(makeUp->isChecked()) {
    _gridResults->setLayout(pageHeightResults->value());
    _curveResults->setLayout(pageHeightResults->value());
  }
  _curveResults->show();
  _gridResults->show();
  QFont f(startBut->font());
  f.setBold(false);
}

void ToolWidget::setRunning(bool r, const QString& message)
{
  TRACE;
  selectBut->setEnabled(!r);
  startBut->setEnabled(!r);
  stopBut->setEnabled(r);
  if(r) {
    if(message.isEmpty()) {
      mainStatus->setText(tr("Running..."));
    } else {
      mainStatus->setText(message);
    }
  } else {
    mainStatus->setText(tr("Not running"));
  }
}

