/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Reader.h"

/*!
  \class Reader Reader.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

Reader::Reader()
{
  _showPlots=0;

  _dispersionTitle=tr("Slowness (s/m)");
  _ellipticityTitle=tr("Ellipticity");
  _noiseTitle=tr("Relative incoherent noise");
  _azimuthTitle=tr("Azimuth (degrees)");
  _powerTitle=tr("Power");

  _dispersionSampling.setMinimum(1.0/3500.0);
  _dispersionSampling.setMaximum(1.0/50.0);
  _dispersionSampling.setScaleType(SamplingParameters::Log);
  _dispersionSampling.setStep(1.015);
  _ellipticitySampling.setMinimum(-90.0);
  _ellipticitySampling.setMaximum(90.0);
  _ellipticitySampling.setStep(1.0);
  _noiseSampling.setMinimum(0.01);
  _noiseSampling.setMaximum(10);
  _noiseSampling.setScaleType(SamplingParameters::Log);
  _noiseSampling.setStep(1.05);
  _azimuthSampling.setMaximum(360.0);
  _azimuthSampling.setStep(5.0);
  _powerSampling.setMaximum(1.0);
  _powerSampling.setCount(200);
  _powerSampling.setScaleType(SamplingParameters::Log);

  memset(_manual.flags, 0, sizeof(int)*5);
  _samples=nullptr;
  _relativeRange=0.2;
  _undefinedEll=true;
  _splitEllSign=false;
  _noiseDev=std::numeric_limits<double>::infinity();
  _slownessDev=0.0;
  _minV=0.0;
  _maxV=std::numeric_limits<double>::infinity();
  _minK=0.0;
  _maxK=std::numeric_limits<double>::infinity();
  _arrayKmin=0.0;
  _arrayKmax=0.0;
}

Reader::~Reader()
{
  delete _samples;
}

void Reader::setRayleigh()
{
  _showPlots=Dispersion;
  _showPlots|=Ellipticity;
  _showPlots|=Noise;
  _showPlots|=Azimuth;
  _showPlots|=Power;
  _undefinedEll=false;
  _splitEllSign=true;
}

bool Reader::setOptions(int& argc, char ** argv)
{
  TRACE;
  // Check arguments
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-R") {
        setRayleigh();
      } else if(arg=="-show-disp") {
        _showPlots|=Dispersion;
      } else if(arg=="-show-ell") {
        _showPlots|=Ellipticity;
      } else if(arg=="-show-noise") {
        _showPlots|=Noise;
      } else if(arg=="-show-az") {
        _showPlots|=Azimuth;
      } else if(arg=="-show-pow") {
        _showPlots|=Power;
      // --------------- DISPERSION
      } else if(arg=="-disp-sampling" || arg=="-y-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_dispersionSampling.setScaleType(argv[i])) {
          App::log(tr("Error setting dispersion sampling\n"));
          return false;
        }
        _manual.dispersion.gridScaleType=true;
      } else if(arg=="-disp-min" || arg=="-y-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.minimum=true;
      } else if(arg=="-disp-max" || arg=="-y-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.maximum=true;
      } else if(arg=="-disp-count" || arg=="-y-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.dispersion.stepCount=true;
        if(_dispersionSampling.count()<=0) {
          App::log(tr("negative or null number of dispersion samples (option -disp-count)\n") );
          return false;
        }
      } else if(arg=="-disp-step" || arg=="-y-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.dispersion.stepCount=true;
        if(_dispersionSampling.step()<=0) {
          App::log(tr("negative or null step for dispersion samples (option -disp-step)\n") );
          return false;
        }
      } else if(arg=="-disp-title" || arg=="-y-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _dispersionTitle=argv[i];
      // --------------- ELLIPTICITY
      } else if(arg=="-ell-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_ellipticitySampling.setScaleType(argv[i])) {
          App::log(tr("Error setting ellipticity sampling\n"));
          return false;
        }
        _manual.ellipticity.gridScaleType=true;
      } else if(arg=="-ell-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.minimum=true;
      } else if(arg=="-ell-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.maximum=true;
      } else if(arg=="-ell-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.ellipticity.stepCount=true;
        if(_ellipticitySampling.count()<=0) {
          App::log(tr("negative or null number of ellipticity samples (option -ell-count)\n") );
          return false;
        }
      } else if(arg=="-ell-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticitySampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.ellipticity.stepCount=true;
        if(_ellipticitySampling.count()<=0) {
          App::log(tr("negative or null number of ellipticity samples (option -ell-count)\n") );
          return false;
        }
      } else if(arg=="-ell-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ellipticityTitle=argv[i];
      // --------------- NOISE
      } else if(arg=="-noise-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_noiseSampling.setScaleType(argv[i])) {
          App::log(tr("Error setting noise sampling\n"));
          return false;
        }
        _manual.noise.gridScaleType=true;
      } else if(arg=="-noise-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.noise.minimum=true;
      } else if(arg=="-noise-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.noise.maximum=true;
      } else if(arg=="-noise-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.noise.stepCount=true;
        if(_noiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -noise-count)\n") );
          return false;
        }
      } else if(arg=="-noise-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.noise.stepCount=true;
        if(_noiseSampling.count()<=0) {
          App::log(tr("negative or null number of noise samples (option -noise-count)\n") );
          return false;
        }
      } else if(arg=="-noise-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseTitle=argv[i];
      // --------------- AZIMUTH
      } else if(arg=="-az-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_azimuthSampling.setScaleType(argv[i])) {
          App::log(tr("Error setting azimuth sampling\n"));
          return false;
        }
        _manual.azimuth.gridScaleType=true;
      } else if(arg=="-az-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.minimum=true;
      } else if(arg=="-az-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.maximum=true;
      } else if(arg=="-az-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.azimuth.stepCount=true;
        if(_azimuthSampling.count()<=0) {
          App::log(tr("negative or null number of azimuth samples (option -az-count)\n") );
          return false;
        }
      } else if(arg=="-az-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.azimuth.stepCount=true;
        if(_azimuthSampling.count()<=0) {
          App::log(tr("negative or null number of azimuth samples (option -az-count)\n") );
          return false;
        }
      } else if(arg=="-az-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _azimuthTitle=argv[i];
      // --------------- POWER
      } else if(arg=="-pow-sampling") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_powerSampling.setScaleType(argv[i])) {
          App::log(tr("Error setting power sampling\n"));
          return false;
        }
        _manual.power.gridScaleType=true;
      } else if(arg=="-pow-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setMinimum(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.minimum=true;
      } else if(arg=="-pow-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setMaximum(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.maximum=true;
      } else if(arg=="-pow-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setCount(CoreApplication::toInt(i, i-1, argv));
        _manual.power.stepCount=true;
        if(_powerSampling.count()<=0) {
          App::log(tr("negative or null number of power samples (option -pow-count)\n") );
          return false;
        }
      } else if(arg=="-pow-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerSampling.setStep(CoreApplication::toDouble(i, i-1, argv));
        _manual.power.stepCount=true;
        if(_powerSampling.count()<=0) {
          App::log(tr("negative or null number of power samples (option -pow-count)\n") );
          return false;
        }
      } else if(arg=="-pow-title") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _powerTitle=argv[i];
      } else if(arg=="-curve-rel-range") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _relativeRange=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-ref-disp-layer") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _referenceDispersionLayer=argv[i];
      } else if(arg=="-ref-ell-layer") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _referenceEllipticityLayer=argv[i];
      } else if(arg=="-undef-ell") {
        _undefinedEll=true;
      } else if(arg=="-no-split-ell-sign") {
        _splitEllSign=false;
      } else if(arg=="-low-noise") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _noiseDev=CoreApplication::toDouble(i, i-1, argv);
        CoreApplication::checkOptionArg(i, argc, argv);
        _slownessDev=CoreApplication::toDouble(i, i-2, argv);
      } else if(arg=="-v-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minV=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-v-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maxV=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-k-min") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minK=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-k-max") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maxK=CoreApplication::toDouble(i, i-1, argv);
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(_showPlots==0) {
    _showPlots=Dispersion;
    _showPlots|=Azimuth;
    _showPlots|=Power;
  }
  return HistogramReader::setOptions(argc, argv);
}

void Reader::helpY(ApplicationHelp * h)
{
  h->addGroup("Dispersion Values", "disp");
  h->addOption("-disp-sampling <SAMPLING>","Defines the slowness sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inversed   regular sampling on an inversed");
  h->addOption("-disp-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-disp-max <MAX>","Maximum of range for slowness axis (default=deduced from data)");
  h->addOption("-disp-step <STEP>","Step between samples along slowness");
  h->addOption("-disp-count <COUNT>","Number of samples along slowness");
  h->addOption("-disp-title <TITLE>","Title for slowness axis");
  h->addGroup("Ellipticity Values", "ell");
  h->addOption("-ell-sampling <SAMPLING>","Defines the ellipticity sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inversed   regular sampling on an inversed");
  h->addOption("-ell-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-ell-max <MAX>","Maximum of range for ellipticity axis (default=deduced from data)");
  h->addOption("-ell-step <STEP>","Step between samples along ellipticity");
  h->addOption("-ell-count <COUNT>","Number of samples along ellipticity");
  h->addOption("-ell-title <TITLE>","Title for ellipticity axis");
  h->addGroup("Noise Values", "ell");
  h->addOption("-noise-sampling <SAMPLING>","Defines the noise sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inversed   regular sampling on an inversed");
  h->addOption("-noise-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-noise-max <MAX>","Maximum of range for noise axis (default=deduced from data)");
  h->addOption("-noise-step <STEP>","Step between samples along noise");
  h->addOption("-noise-count <COUNT>","Number of samples along noise");
  h->addOption("-noise-title <TITLE>","Title for noise axis");
  h->addGroup("Azimuth Values", "az");
  h->addOption("-az-sampling <SAMPLING>","Defines the azimuth sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inversed   regular sampling on an inversed");
  h->addOption("-az-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-az-max <MAX>","Maximum of range for azimuth axis (default=deduced from data)");
  h->addOption("-az-step <STEP>","Step between samples along azimuth");
  h->addOption("-az-count <COUNT>","Number of samples along azimuth");
  h->addOption("-az-title <TITLE>","Title for azimuth axis");
  h->addGroup("Power Values", "pow");
  h->addOption("-pow-sampling <SAMPLING>","Defines the power sampling type:\n"
                                        "  linear     regular sampling\n"
                                        "  log        regular sampling on a log scale\n"
                                        "  inversed   regular sampling on an inversed");
  h->addOption("-pow-min <MIN>","Minimum of range for slownness axis (default=deduced from data)");
  h->addOption("-pow-max <MAX>","Maximum of range for power axis (default=deduced from data)");
  h->addOption("-pow-step <STEP>","Step between samples along power");
  h->addOption("-pow-count <COUNT>","Number of samples along power");
  h->addOption("-pow-title <TITLE>","Title for power axis");
}

/*!
  Grab some processing parameter values.
*/
bool Reader::parse(QTextStream& s)
{
  TRACE;
  QString line;
  bool maxFormatHeader=false;
  do {
    line=s.readLine();
    if(!maxFormatHeader && line.startsWith("# MAX FORMAT RELEASE")) {
      maxFormatHeader=true;
    }
    if(line.startsWith("K_MIN")) {
      _arrayKmin=line.section("=", 1, -1).toDouble();
    } else if(line.startsWith("K_MAX")) {
      _arrayKmax=line.section("=", 1, -1).toDouble();
    } else if(line.startsWith("PROCESS_TYPE")) {
      if(line.section("=", 1, -1)=="RTBF") {
        setRayleigh();
      }
    } else if(line.startsWith("# abs_time frequency slowness azimuth ")) {
      //if(!configureFKMaxFile(line)) {
      //  return false;
      //}
      maxFormatHeader=false;
    }
  } while (maxFormatHeader || line.startsWith("#"));

  // Effectively load samples, last line is provided because we may reading
  // stdin where seek() is not supported.
  return loadSamples(s, line);
}

/*!

*/
bool Reader::loadSamples(QTextStream& s, const QString& lastLine)
{
  TRACE;
  ConsoleProgress progress;
  progress.setCaption(tr("Loading"));
  bool ok=true;
  Sample sample;
  QString line;
  if(File::isDataLine(lastLine)) {
    if(sample.read(lastLine)) {
      _samples->append(sample);
    } else {
      ok=false;
    }
  }
  while(!s.atEnd()) {
    if(File::readLine(s, line)) {
      if(sample.read(line)) {
        _samples->append(sample);
      } else {
        ok=false;
      }
    }
    progress.setValue(_samples->count());
  }
  progress.end(_samples->count());
  return ok;
}

/*!
  Computes default limits from samples
*/
bool Reader::setDefaultLimits()
{
  TRACE;
  if(_samples->isEmpty()) {
    App::log(tr("No samples available\n") );
    return false;
  }
  QVector<Sample>::iterator it;
  QVector<double> x;
  for(it=_samples->begin(); it!=_samples->end(); it++) {
    x.append(it->frequency());
  }
  setDefaultXLimits(x);

  double m;
  if(!_manual.dispersion.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->slowness()<m) {
        m=it->slowness();
      }
    }
    _dispersionSampling.setMinimum(m);
  }
  if(!_manual.dispersion.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->slowness()>m) {
        m=it->slowness();
      }
    }
    _dispersionSampling.setMaximum(m);
  }

  if(!_dispersionSampling.isValidRange(true)) {
    App::log(tr("Error while checking dispersion axis\n"));
    return false;
  }
  _dispersionSampling.includeLimits();

  if(!_manual.ellipticity.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->ellipticity()<m) {
        m=it->ellipticity();
      }
    }
    _ellipticitySampling.setMinimum(m);
  }
  if(!_manual.ellipticity.maximum) {
    m=-std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->ellipticity()>m) {
        m=it->ellipticity();
      }
    }
    _ellipticitySampling.setMaximum(m);
  }

  if(!_ellipticitySampling.isValidRange(true)) {
    App::log(tr("Error while checking ellipticity axis\n"));
    return false;
  }
  _ellipticitySampling.includeLimits();

  if(!_manual.noise.minimum) {
    m=std::numeric_limits<double>::infinity();
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->noise()<m) {
        m=it->noise();
      }
    }
    _noiseSampling.setMinimum(m);
  }
  if(!_manual.noise.maximum) {
    if(_noiseSampling.minimum()<50.0) {
      _noiseSampling.setMaximum(50.0);
    } else {
       _noiseSampling.setRange(_noiseSampling.minimum(), _noiseSampling.minimum());
    }
  }

  if(!_noiseSampling.isValidRange(true)) {
    App::log(tr("Back to linear scale for noise axis\n"));
    _noiseSampling.setScaleType(SamplingParameters::Linear);
  }
  _noiseSampling.includeLimits();

  if(!_manual.azimuth.minimum) {
    m=360.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->azimuth()<m) {
        m=it->azimuth();
      }
    }
    _azimuthSampling.setMinimum(m);
  }
  if(!_manual.azimuth.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->azimuth()>m) {
        m=it->azimuth();
      }
    }
    _azimuthSampling.setMaximum(m);
  }

  if(!_azimuthSampling.isValidRange(true)) {
    App::log(tr("Error while checking azimuth axis\n"));
    return false;
  }
  _azimuthSampling.includeLimits();

  if(!_manual.power.maximum) {
    m=0.0;
    for(it=_samples->begin(); it!=_samples->end(); it++) {
      if(it->power()>m) {
        m=it->power();
      }
    }
    _powerSampling.setMaximum(m);
  }
  if(!_manual.power.minimum) {
    _powerSampling.setMinimum(_powerSampling.maximum()*0.001);
  }

  if(!_powerSampling.isValidRange(true)) {
    App::log(tr("Error while checking power axis\n"));
    return false;
  }
  _powerSampling.includeLimits();

  return true;
}

Histogram2D * Reader::histogram(const SamplingParameters& ySampling) const
{
  TRACE;
  Histogram2D * h=HistogramReader::histogram(ySampling);
  if(h) {
    h->setXTitle(tr("Frequency (Hz)"));
  }
  return h;
}
