/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiWave.h>

#include "ModeWidgets.h"

/*!
  \class ModeWidgets ModeWidgets.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
ModeWidgets::ModeWidgets(QObject * parent)
  : QObject(parent)
{
  TRACE;
  _dispersion=nullptr;
  _ellipticity=nullptr;
  _noise=nullptr;
  _azimuth=nullptr;
  _power=nullptr;
}

/*!
  Description of destructor still missing
*/
ModeWidgets::~ModeWidgets()
{
  TRACE;
}

void ModeWidgets::setHistogram(IrregularGrid2DPlot * layer, Histogram2D * hist, bool normalized)
{
  TRACE;
  if(hist) {
    if(normalized) {
      hist->normalizeArea(YAxis);
    }
    layer->setGrid(*hist);
    if(!hist->xTitle().isNull()) {
      layer->graph()->xAxis()->setTitle(hist->xTitle());
    }
    if(!hist->yTitle().isNull()) {
      layer->graph()->yAxis()->setTitle(hist->yTitle());
    }
    if(!hist->yTitleInversedScale().isNull()) {
      layer->graph()->yAxis()->setTitleInversedScale(hist->yTitleInversedScale());
    }
    layer->graph()->xAxis()->setScaleType(hist->xScaleType());
    layer->graph()->yAxis()->setScaleType(hist->yScaleType());
    delete hist;
  }
}

void ModeWidgets::setComments(const QString &cmt)
{
  TRACE;
  if(_dispersion) _dispersion->setComments(cmt);
  if(_ellipticity) _ellipticity->setComments(cmt);
  if(_noise) _noise->setComments(cmt);
  if(_azimuth) _azimuth->setComments(cmt);
  if(_power) _power->setComments(cmt);
}

void ModeWidgets::autoPickDispersion(LineLayer * layer, Histogram2D * hist)
{
  layer->clear();
  Curve<RealStatisticalPoint> all=hist->pickAll(4, 0.1, 0.2);
  layer->addLine(Pen(Qt::NoPen), Symbol(Symbol::Circle, 0.5, Pen(Qt::red), Brush(Qt::SolidPattern)));
  static_cast<RealStatisticalLine *>(layer->line(0))->curve()=all;

  QList<Curve<RealStatisticalPoint>> curves=all.split(1.2, LogScale, 0.1, 5);
  int n=curves.count();
  for(int i=0; i<n; i++) {
    layer->addLine(Pen(Qt::black), Symbol(Symbol::Circle, 0.8, Pen(Qt::black), Brush(Qt::SolidPattern)));
    Curve<RealStatisticalPoint>& c=static_cast<RealStatisticalLine *>(layer->line(i+1))->curve();
    c=curves.at(i);
  }
}

void ModeWidgets::autoPickEllipticity(LineLayer * layer, Histogram2D * hist)
{
  layer->clear();
  Curve<RealStatisticalPoint> all=hist->pickAll(3, 5.0, 10.0); // in deg.
  layer->addLine(Pen(Qt::NoPen), Symbol(Symbol::Circle, 0.5, Pen(Qt::red), Brush(Qt::SolidPattern)));
  static_cast<RealStatisticalLine *>(layer->line(0))->curve()=all;

  QList<Curve<RealStatisticalPoint>> curves=all.split(1.2, LogScale, 0.1, 5);
  int n=curves.count();
  for(int i=0; i<n; i++) {
    layer->addLine(Pen(Qt::black), Symbol(Symbol::Circle, 0.8, Pen(Qt::black), Brush(Qt::SolidPattern)));
    Curve<RealStatisticalPoint>& c=static_cast<RealStatisticalLine *>(layer->line(i+1))->curve();
    c=curves.at(i);
  }
}

void ModeWidgets::setSamples(const Samples& s, const Reader& param)
{
  TRACE;
  if(_dispersion) {
    Histogram2D * hist=dispersionHistogram(s, param);
    autoPickDispersion(_dispersionPick, hist);
    setHistogram(_dispersion, hist, param.isNormalized());
  }
  if(_ellipticity) {
    Histogram2D * hist=ellipticityHistogram(s, param);
    autoPickEllipticity(_ellipticityPick, hist);
    setHistogram(_ellipticity, hist, param.isNormalized());
  }
  if(_noise) {
    setHistogram(_noise, noiseHistogram(s, param), param.isNormalized());
  }
  if(_azimuth) {
    setHistogram(_azimuth, azimuthHistogram(s, param), param.isNormalized());
  }
  if(_power) {
    setHistogram(_power, powerHistogram(s, param), param.isNormalized());
  }
}

void ModeWidgets::addWidgets(GraphicSheet * sheet, const Reader& param)
{
  TRACE;
  AxisWindow * w;
  ColorMap map;
  double width=8.0;
  double rightPos=0.5+9.5*(sheet->objects().count()/5+1);
  double topPos=0.5;
  map.generateColorScale(20, ColorPalette::McNamesClip, true);
  map.setWhiteTransparent(true);

  if(param.showPlots() & Reader::Dispersion) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Dispersion");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    _dispersion=new IrregularGrid2DPlot(w);
    _dispersion->setObjectName("Histogram");
    _dispersion->setSmooth(false);
    _dispersion->setColorMap(map);
    _dispersionPick=new LineLayer(w);
    _dispersionPick->setObjectName("Pick");
    _dispersionPick->setReferenceLine(new RealStatisticalLine);
    _dispersionPick->setLogErrorBar(true);
    _dispersionPick->setOpacity(0.5);
    DispersionLimitLayer * limits=new DispersionLimitLayer(w);
    limits->setObjectName("FK dispersion limits");
    limits->addArrayLimits();
    limits->setFrequencySampling(*param.xSampling());
    limits->setArrayKmin(param.arrayKmin());
    limits->setArrayKmax(param.arrayKmax());
    if(!param.referenceDispersionLayer().isEmpty()) {
      w->graphContent()->appendLayers(param.referenceDispersionLayer());
    }
    w->updateExternalGeometry();
    sheet->showObject(w);
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::Ellipticity) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Ellipticity");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    _ellipticity=new IrregularGrid2DPlot(w);
    _ellipticity->setObjectName("Histogram");
    _ellipticity->setSmooth(false);
    _ellipticity->setColorMap(map);
    _ellipticityPick=new LineLayer(w);
    _ellipticityPick->setObjectName("Pick");
    _ellipticityPick->setReferenceLine(new RealStatisticalLine);
    _ellipticityPick->setOpacity(0.5);
    if(!param.referenceEllipticityLayer().isEmpty()) {
      w->graphContent()->appendLayers(param.referenceEllipticityLayer());
    }
    w->updateExternalGeometry();
    sheet->showObject(w);
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::Noise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Noise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    _noise=new IrregularGrid2DPlot(w);
    _noise->setObjectName("Histogram");
    _noise->setSmooth(false);
    _noise->setColorMap(map);
    w->updateExternalGeometry();
    sheet->showObject(w);
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::Azimuth) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Azimuth");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    _azimuth=new IrregularGrid2DPlot(w);
    _azimuth->setObjectName("Histogram");
    _azimuth->setSmooth(false);
    _azimuth->setColorMap(map);
    w->updateExternalGeometry();
    sheet->showObject(w);
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::Power) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Power");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    _power=new IrregularGrid2DPlot(w);
    _power->setObjectName("Histogram");
    _power->setSmooth(false);
    _power->setColorMap(map);
    w->updateExternalGeometry();
    sheet->showObject(w);
  }
}

Histogram2D * ModeWidgets::dispersionHistogram(const Samples& s, const Reader& param)
{
  TRACE;
  App::log(tr("Dispersion histogram\n"));
  Histogram2D * h=param.histogram(param.dispersionSampling());
  if(!h) {
    return nullptr;
  }
  // TODO support alternative ways of plotting dispersion (same for ellipticity)
  h->setYTitle(param.dispersionTitle());
  h->setYTitleInversedScale(tr("Velocity (m/s)"));
  h->setYScaleType(Scale::InversedLog);
  h->setSamples(s.dispersionSamples());
  return h;
}

Histogram2D * ModeWidgets::ellipticityHistogram(const Samples& s, const Reader& param)
{
  TRACE;
  App::log(tr("Ellipticity histogram\n"));
  Histogram2D * h=param.histogram(param.ellipticitySampling());
  if(!h) {
    return nullptr;
  }
  h->setYTitle(param.ellipticityTitle());
  h->setYScaleType(Scale::Linear);
  h->setSamples(s.ellipticitySamples());
  return h;
}

Histogram2D * ModeWidgets::noiseHistogram(const Samples& s, const Reader& param)
{
  TRACE;
  App::log(tr("Noise histogram\n"));
  Histogram2D * h=param.histogram(param.noiseSampling());
  if(!h) {
    return nullptr;
  }
  h->setYTitle(param.noiseTitle());
  h->setYScaleType(Scale::Log);
  h->setSamples(s.noiseSamples());
  return h;
}

Histogram2D * ModeWidgets::azimuthHistogram(const Samples& s, const Reader& param)
{
  TRACE;
  App::log(tr("Azimuth histogram\n"));
  Histogram2D * h=param.histogram(param.azimuthSampling());
  if(!h) {
    return nullptr;
  }
  h->setYTitle(param.azimuthTitle());
  h->setYScaleType(Scale::Linear);
  h->setSamples(s.azimuthSamples());
  return h;
}

Histogram2D * ModeWidgets::powerHistogram(const Samples& s, const Reader& param)
{
  TRACE;
  App::log(tr("Power histogram\n"));
  Histogram2D * h=param.histogram(param.powerSampling());
  if(!h) {
    return nullptr;
  }
  h->setYTitle(param.powerTitle());
  h->setYScaleType(Scale::Log);
  h->setSamples(s.powerSamples());
  return h;
}

void ModeWidgets::setLimits(bool normalize)
{
  TRACE;
  if(_dispersion) setLimits(_dispersion, normalize);
  if(_ellipticity) setLimits(_ellipticity, normalize);
  if(_noise) setLimits(_noise, normalize);
  if(_azimuth) setLimits(_azimuth, normalize);
  if(_power) setLimits(_power, normalize);
}

void ModeWidgets::setLimits(IrregularGrid2DPlot * layer, bool normalize)
{
  TRACE;
  GraphContent * gc=layer->graphContent();
  AxisWindow * w=gc->graph();
  Rect r=layer->boundingRect();
  r.enlarge(0.01, gc->scaleX().sampling(), gc->scaleY().sampling());
  w->xAxis()->setRange(r.x1(), r.x2());
  w->yAxis()->setRange(r.y1(), r.y2());

  ColorMap pal=layer->colorMap();
  double m=layer->grid().maximumValue();
  if(normalize) {
    pal.setValues(0.0001*m, m, SamplingParameters::Log);
  } else {
    pal.generateColorScale(qRound(m+1.0), ColorPalette::McNamesClip, true);
    pal.setValues(0.5, m-0.5, SamplingParameters::Linear);
  }
  layer->setColorMap(pal);
  w->updateGeometry(); // TopRight anchor imply that position can change when changing axis limits
}

void ModeWidgets::setColorMap(const ColorMap& map)
{
  if(_dispersion) _dispersion->setColorMap(map);
  if(_ellipticity) _ellipticity->setColorMap(map);
  if(_noise) _noise->setColorMap(map);
  if(_azimuth) _azimuth->setColorMap(map);
  if(_power) _power->setColorMap(map);
}
