/***************************************************************************
**
**  This file is part of SciFigs.
**
**  SciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  SciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-01-22
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PageSizeProperties.h"

namespace SciFigs {

  QList<QPair<QString, QPageSize::PageSizeId>> PageSizeProperties::_paperSizes;

  /*!
    \class PageSizeProperties PageSizeProperties.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  PageSizeProperties::PageSizeProperties(QWidget * parent)
    : PropertyWidget(parent)
  {
    TRACE;
    setupUi(this);
    setPaperSizes();

    // Create properties
    addProperty(PaperSize, paperSizeEdit, paperSizeLabel);
    addProperty(Unit, unitEdit, unitLabel);
    addProperty(Width, widthEdit, widthLabel);
    addProperty(Height, heightEdit, heightLabel);
    addProperty(Orientation, orientationEdit, orientationLabel);

    widthEdit->setMaximum(std::numeric_limits<double>::infinity());
    heightEdit->setMaximum(std::numeric_limits<double>::infinity());
    _unit=QPageSize::Millimeter;
    paperSizeEdit->setCurrentIndex(paperSize2item(QPageSize::A4));
    widthEdit->setSuffix(tr(" mm"));
    heightEdit->setSuffix(tr(" mm"));
  }

  PropertyValue::WidgetType PageSizeProperties::determineCustomWidgetType(int pid, QWidget *, QWidget * )
  {
    TRACE;
    switch(pid) {
    case PaperSize:
    case Unit:
      return PropertyValue::Custom0;
    default:
      return PropertyValue::Unknown;
    }
  }

  bool PageSizeProperties::connectCustomWidget(PropertyValue & p)
  {
    TRACE;
    if(p.widgetType()==PropertyValue::Custom0) {
      // No connection here, already connected through setupUi()
      return true;
    } else {
      return false;
    }
  }

  bool PageSizeProperties::setCustomWidget(PropertyValue & p)
  {
    TRACE;
    if(p.widgetType()==PropertyValue::Custom0) {
      static_cast<QComboBox *>(p.widget())->setCurrentIndex(p.value().toInt());
      return true;
    } else {
      return false;
    }
  }

  QVariant PageSizeProperties::customWidgetValue(PropertyValue & p)
  {
    TRACE;
    if(p.widgetType()==PropertyValue::Custom0) {
      return static_cast<QComboBox *>(p.widget())->currentIndex();
    } else {
      return QVariant();
    }
  }

  QPageSize::PageSizeId PageSizeProperties::item2paperSize(int index)
  {
    TRACE;
    return _paperSizes.at(index).second;
  }

  int PageSizeProperties::paperSize2item(QPageSize::PageSizeId id)
  {
    TRACE;
    for(int i=_paperSizes.count()-1; i>=0; i--) {
      if(_paperSizes.at(i).second==id) {
        return i;
      }
    }
    return 0;
  }

  QPageSize::Unit PageSizeProperties::item2unit(int index)
  {
    TRACE;
    switch(index) {
    case 1:
      return QPageSize::Inch;
    case 2:
      return QPageSize::Point;
    default:
      return QPageSize::Millimeter;
    }
  }

  int PageSizeProperties::unit2item(QPageSize::Unit unit)
  {
    TRACE;
    switch(unit) {
    case QPageSize::Inch:
      return 1;
    case QPageSize::Point:
      return 2;
    default:
      return 0;
    }
  }

  QPageLayout::Orientation PageSizeProperties::item2orientation(int index)
  {
    TRACE;
    switch(index) {
    case 1:
      return QPageLayout::Landscape;
    default:
      return QPageLayout::Portrait;
    }
  }

  int PageSizeProperties::orientation2item(QPageLayout::Orientation orientation)
  {
    TRACE;
    switch(orientation) {
    case QPageLayout::Landscape:
      return 1;
    default:
      return 0;
    }
  }

  void PageSizeProperties::setPaperSizes()
  {
    TRACE;
    if(_paperSizes.isEmpty()) {
      for(int i=0; i<=QPageSize::LastPageSize; i++) {
        QPageSize ps(static_cast<QPageSize::PageSizeId>(i));
        QString n=ps.name();
        if(!n.isEmpty()) {
          _paperSizes.append(QPair<QString, QPageSize::PageSizeId>(n, ps.id()));
        }
      }
      QCollator collator;
      collator.setNumericMode(true);
      std::sort(_paperSizes.begin(), _paperSizes.end(),
          [&collator](const QPair<QString, QPageSize::PageSizeId>& ps1,
                      const QPair<QString, QPageSize::PageSizeId>& ps2)
          {
              return collator.compare(ps1.first, ps2.first)<0;
          });
      _paperSizes.prepend(QPair<QString, QPageSize::PageSizeId>(tr("Custom"), QPageSize::Custom));
    }
    int n=_paperSizes.count();
    paperSizeEdit->blockSignals(true);
    for(int i=0; i<n; i++) {
      paperSizeEdit->addItem(_paperSizes.at(i).first);
    }
    paperSizeEdit->blockSignals(false);
  }

  void PageSizeProperties::on_unitEdit_currentIndexChanged(int index)
  {
    TRACE;
    QPageSize ps(_paperSizes.at(paperSizeEdit->currentIndex()).second);
    double w, h;
    // Gets size and converts it to millimeters
    if(ps.id()==QPageSize::Custom) {
      w=widthEdit->value();
      h=heightEdit->value();
    } else {
      if(item2orientation(orientationEdit->currentIndex())==QPageLayout::Portrait) {
        w=ps.definitionSize().width();
        h=ps.definitionSize().height();
      } else {
        w=ps.definitionSize().height();
        h=ps.definitionSize().width();
      }
      _unit=ps.definitionUnits();
    }
    switch(_unit) {
    case QPageSize::Inch:
      w*=25.4;
      h*=25.4;
      break;
    case QPageSize::Point:
      w*=25.4/72.0;
      h*=25.4/72.0;
      break;
    default:
      break;
    }
    setProperty(Unit, index);
    _unit=item2unit(index);
    switch(_unit) {
    case QPageSize::Inch:
      widthEdit->setDecimals(2);
      heightEdit->setDecimals(2);
      widthEdit->setSuffix(tr(" in"));
      heightEdit->setSuffix(tr(" in"));
      widthEdit->setValue(w/25.4);
      heightEdit->setValue(h/25.4);
      break;
    case QPageSize::Point:
      widthEdit->setDecimals(0);
      heightEdit->setDecimals(0);
      widthEdit->setSuffix(tr(" pt"));
      heightEdit->setSuffix(tr(" pt"));
      widthEdit->setValue(w/(25.4/72.0));
      heightEdit->setValue(h/(25.4/72.0));
      break;
    default:
      widthEdit->setDecimals(1);
      heightEdit->setDecimals(1);
      widthEdit->setSuffix(tr(" mm"));
      heightEdit->setSuffix(tr(" mm"));
      widthEdit->setValue(w);
      heightEdit->setValue(h);
      break;
    }
  }

  void PageSizeProperties::on_paperSizeEdit_currentIndexChanged(int index)
  {
    TRACE;
    QPageSize ps(_paperSizes.at(index).second);
    bool customSizeEnabled=(ps.id()==QPageSize::Custom);
    widthEdit->setEnabled(customSizeEnabled);
    heightEdit->setEnabled(customSizeEnabled);
    setProperty(PaperSize, index);
    if(!customSizeEnabled) {
      if(item2orientation(orientationEdit->currentIndex())==QPageLayout::Portrait) {
        widthEdit->setValue(ps.definitionSize().width());
        heightEdit->setValue(ps.definitionSize().height());
      } else {
        widthEdit->setValue(ps.definitionSize().height());
        heightEdit->setValue(ps.definitionSize().width());
      }
      _unit=ps.definitionUnits();
      unitEdit->setCurrentIndex(unit2item(_unit));
    }
  }

} // namespace SciFigs

