/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-07-04
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "NameLineLayerProperties.h"
#include "NameLineLayer.h"
#include "NameLineItem.h"
#include "NameLineDelegate.h"
#include "AxisWindow.h"

namespace SciFigs {

NameLineLayerProperties::NameLineLayerProperties(QWidget * parent)
    : PropertyWidget(parent)
{
  TRACE;
  setupUi(this);

  addProperty(ShowNames, showNames);
  addProperty(Font, fontEdit);

  NameLineItem * model=new NameLineItem(this);
  lineTable->setModel(model);
  NameLineDelegate * del=new NameLineDelegate(this);
  lineTable->setItemDelegate(del);
  lineTable->setSelectionBehavior(QAbstractItemView::SelectItems);
  lineTable->setSelectionMode(QAbstractItemView::ExtendedSelection);
  lineTable->setEditTriggers(QAbstractItemView::AllEditTriggers);
  lineTable->resizeColumnsToContents();
  Settings::columnWidth(lineTable, "NameLineLayerProperties");

  connect(del, SIGNAL(dataChanged()), this, SLOT(applyAllSelected()) );
}

/*!
  Description of destructor still missing
*/
NameLineLayerProperties::~NameLineLayerProperties()
{
  TRACE;
  Settings::setColumnWidth(lineTable, "NameLineLayerProperties");
}

PropertyValue::WidgetType NameLineLayerProperties::determineCustomWidgetType(int pid, QWidget *, QWidget * )
{
  TRACE;
  if(pid==Font) {
    return PropertyValue::Custom0;
  } else {
    return PropertyValue::Unknown;
  }
}

bool NameLineLayerProperties::connectCustomWidget(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    return true;
  } else {
    return false;
  }
}

bool NameLineLayerProperties::setCustomWidget(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    QFont f;
    if(f.fromString(p.value().toString()) ) {
      _font=f;
    }
    return true;
  } else {
    return false;
  }
}

QVariant NameLineLayerProperties::customWidgetValue(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    return _font.toString();
  } else {
    return QVariant();
  }
}

void NameLineLayerProperties::on_fontEdit_clicked()
{
  TRACE;
  bool ok;
  QFont f=QFontDialog::getFont(&ok, _font, this);
  if(ok) {
    _font=f;
    propertyValue(Font)->touched();
  }
}

void NameLineLayerProperties::addLayer(NameLineLayer * layer)
{
  TRACE;
  if(_layerSet.contains(layer)) return;
  _layerSet.insert(layer);
  setLayerList();
  on_currentLayerCombo_currentIndexChanged(0);
}

void NameLineLayerProperties::removeLayer(NameLineLayer * layer)
{
  TRACE;
  if(!_layerSet.contains(layer)) return;
  _layerSet.remove(layer);
  setLayerList();
  on_currentLayerCombo_currentIndexChanged(0);
}

bool NameLineLayerProperties::layerLessThan(const NameLineLayer * l1, const NameLineLayer * l2)
{
  QString l1Name, l2Name;
  if(l1->graph()->objectName().isEmpty())
    l1Name="unamed_graph::" + l1->objectName();
  else
    l1Name=l1->graph()->objectName() + "::" + l1->objectName();
  if(l2->graph()->objectName().isEmpty())
    l2Name="unamed_graph::" + l2->objectName();
  else
    l2Name=l2->graph()->objectName() + "::" + l2->objectName();
  return l1Name<l2Name;
}

void NameLineLayerProperties::setLayerList()
{
  TRACE;
  _layerList.clear();
  for(QSet<NameLineLayer *>::iterator it=_layerSet.begin();it!=_layerSet.end();it++) {
    _layerList.append(*it);
  }
  std::sort(_layerList.begin(), _layerList.end(), layerLessThan);
  currentLayerCombo->clear();
  for(QList<NameLineLayer *>::iterator it=_layerList.begin();it!=_layerList.end();it++) {
    LineLayer * layer=*it;
    if(layer->graph()->objectName().isEmpty())
      currentLayerCombo->addItem( "unamed_graph::" + layer->objectName());
    else
      currentLayerCombo->addItem(layer->graph()->objectName() + "::" + layer->objectName());
  }
}

NameLineLayer * NameLineLayerProperties::currentLayer() const
{
  TRACE;
  int i=currentLayerCombo->currentIndex();
  if(i<_layerList.count() && i>=0) {
    return _layerList[i];
  } else {
    return 0;
  }
}

void NameLineLayerProperties::on_currentLayerCombo_currentIndexChanged(int)
{
  TRACE;
  NameLineLayer * selectedLayer=currentLayer();
  NameLineItem * item=qobject_cast<NameLineItem *>(lineTable->model());
  item->setLayer(selectedLayer);
}


void NameLineLayerProperties::applyAllSelected()
{
  TRACE;
  QAbstractItemModel& model=* lineTable->model();
  QModelIndex curIndex=lineTable->currentIndex();
  QModelIndexList l=lineTable->selectionModel()->selectedRows();
  if(l.count()>1) {
    QVariant val=model.data(curIndex);
    QModelIndex index;
    foreach(index, l) {
      model.setData(model.index(index.row(), curIndex.column()), val);
    }
  }
}

} // namespace SciFigs
