/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-01-11
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Dialog.h"

namespace QGpGuiTools {

  /*!
    \class Dialog Dialog.h
    \brief Dialog class

  */

  Dialog::Dialog(QWidget * parent, Qt::WindowFlags f)
    : QDialog(parent, f)
  {
    TRACE;
    _vboxLayout=0;
    _buttons=None;
  }

  void Dialog::resizeEvent(QResizeEvent * e)
  {
    TRACE;
    WidgetRect::record(this);
    QDialog::resizeEvent(e);
  }

  void Dialog::moveEvent(QMoveEvent * e)
  {
    TRACE;
    WidgetRect::record(this);
    QDialog::moveEvent(e);
  }

  void Dialog::closeEvent(QCloseEvent * e)
  {
    TRACE;
    if(_buttons==TitleClose) {
      emit(accept());
      e->accept();
    } else {
      QDialog::closeEvent(e);
    }
  }

  /*!
    Shows any number of widgets (vertical layout) in a modal dialog box with bottom buttons.

    \sa addButtons(), setMainWidget()
  */
  void Dialog::addWidget(QWidget * w)
  {
    TRACE;
    if(!_vboxLayout) {
      _vboxLayout=new QVBoxLayout(this);
      _vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));
    }
    _vboxLayout->addWidget(w);

    if(windowTitle().isEmpty()) setWindowTitle(w->windowTitle());
  }

  /*!
    Shows any number of widgets (vertical layout) in a modal dialog box with bottom buttons.
    At least one call to addWidget is required by calling this function.

    \sa addWidget(), setMainWidget()
  */
  void Dialog::addButtons(Buttons b)
  {
    TRACE;
    ASSERT(_vboxLayout);

    QDialogButtonBox * buttonBox=0;
    switch(b) {
    case OkCancel:
      buttonBox=new QDialogButtonBox(this);
      buttonBox->addButton(QDialogButtonBox::Ok);
      buttonBox->addButton(QDialogButtonBox::Cancel);
      connect (buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
      connect (buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
      _vboxLayout->addWidget(buttonBox);
      resize(minimumSizeHint());
      break;
    case Close:
      buttonBox=new QDialogButtonBox(this);
      buttonBox->addButton(QDialogButtonBox::Close);
      connect (buttonBox, SIGNAL(rejected()), this, SLOT(accept()));
      _vboxLayout->addWidget(buttonBox);
      resize(minimumSizeHint());
      break;
    case TitleClose:
    case None:
      break;
    }
    _buttons=b;
  }

  /*!
    Shows any widget in a modal dialog box with bottom buttons.
    Convenient function for dialogs containing only one widget.
  */
  void Dialog::setMainWidget(QWidget * w, Buttons b)
  {
    TRACE;
    addWidget(w);
    switch(b) {
    case None:
    case TitleClose:
      _buttons=b;
      break;
    case OkCancel:
    case Close:
      addButtons(b);
      break;
    }
  }

#if(QT_VERSION < QT_VERSION_CHECK(5, 0, 0))
  /*!

  */
  void Dialog::setWindowTitle(const QString& t)
  {
    QWidget::setWindowTitle(t);
    emit windowTitleChangedQt4(t);
  }
#endif

  void FileDialog::resizeEvent(QResizeEvent * e)
  {
    TRACE;
    WidgetRect::record(this);
    QFileDialog::resizeEvent(e);
  }

  void FileDialog::moveEvent(QMoveEvent * e)
  {
    TRACE;
    WidgetRect::record(this);
    QFileDialog::moveEvent(e);
  }

} // namespace QGpGuiTools
