/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-10-31
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef CURVEBROWSER_H
#define CURVEBROWSER_H

#include <QGpGuiTools.h>
#include <QGpCoreMath.h>
#include <SciFigs.h>

#include "ui_CurveBrowser.h"
#include "QGpGuiMathDLLExport.h"

namespace QGpGuiMath {

  class CurveProperties;
  class CurvePlotProxy;

  class QGPGUIMATH_EXPORT CurveBrowser : public QWidget, protected Ui::CurveBrowser
  {
    Q_OBJECT
  public:
    CurveBrowser(QWidget * parent=nullptr);
    ~CurveBrowser();

    void setProxy(CurveProxy *proxy);
    CurveProxy * proxy() {return _proxy;}

    void setPlotProxy(CurvePlotProxy * proxy);
    CurvePlotProxy * plotProxy() {return _plotProxy;}

    CurveProperties * find(const QString& name) const;
    CurveProperties * properties(AbstractLine * line) const;
    QList<CurveProperties *> curves() const;
    void dock(QWidget * w);

    QAction * addCurveAction(QString title, QString toolTip, bool editCurve);
    void setEditable(bool e);

    void initLayer(LineLayer * curveLayer);
    void setCurrentLayer(LineLayer * curveLayer);
    LineLayer * currentLayer() const {return _currentLayer;}
    CurveProperties * addLine(AbstractLine * line);

    int count() const;
    int currentLine() const;
    bool loadMultiColumns(const QString& fileName, bool lastFile, ColumnTextParser *& parser);
    virtual bool loadMultiColumns(const ColumnTextParser * parser, const QString& fileName);
    void setLegend(const Legend& l);
  public slots:
    virtual void load();
    void clear();
    void legend();
    void average();
    void validate();
    virtual void setLimits();
    void setCurveName(QString n);
    void showLog();
    void save();
    void remove();
    void showCurves();
    void hideCurves();
    void resample();
    void cut();
    void smooth();
    void split();
  private slots:
    void on_curveTabs_tabCloseRequested(int index);
    void on_loadBut_clicked() {load();}
    void on_legendBut_clicked() {legend();}
    void on_clearBut_clicked();
    void newPickedLine(AbstractLine * line);
    void beginNewPickedPoint(AbstractLine * line);
    void endNewPickedPoint(AbstractLine * line);
    void commitLegend();
  signals:
    void curveModified();
  protected:
    bool loadMultiColumns(LineLayer * layer, ColumnTextIterator& it,
                          const QString& fileName, int& nCurves);
    virtual CurveProperties * createProperties();
    static void curveRange(const QList<CurveProperties *> &list, double& min, double& max);
    QList<CurveProperties *> selectCurves();
    void closeEvent(QCloseEvent * e);
    void takeLine(int index);
    void removeLine(int index);
  private:
    LineLayer * _currentLayer;
    int _currentCurve;
    CurveProxy * _proxy;
    CurvePlotProxy * _plotProxy;
    LegendProperties * _legendEditor;
    bool _editable;
  };

} // namespace QGpGuiMath

#endif // CURVEBROWSER_H
