/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-05-20
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MAGNETOTELLURICPOINTOPTIONS_H
#define MAGNETOTELLURICPOINTOPTIONS_H

#include <QGpCoreMath.h>

#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class QGPCOREWAVE_EXPORT MagnetoTelluricPointOptions : public AbstractComplexPointOptions
{
public:
  enum OutputMode {ApparentResistivity, AbsoluteValue, PhaseDegrees, PhaseRadians, Real, Imaginary};

  MagnetoTelluricPointOptions(OutputMode m=ApparentResistivity) : AbstractComplexPointOptions() {_mode=m;}
  MagnetoTelluricPointOptions(const MagnetoTelluricPointOptions& o) : AbstractComplexPointOptions() {_mode=o._mode;}

  void setMode(OutputMode m) {_mode=m;}
  OutputMode mode() const {return _mode;}

  virtual double toDouble(double x, const Complex& c) const;
  virtual void fromDouble(double x, Complex& c, double v) const;

  static inline double toAbsoluteValue(double frequency, double apparentResistivity);
  static inline double toApparentResistivity(double frequency, double absoluteValue);
private:
  OutputMode _mode;
};

#define MAGNETIC_CONSTANT (4e-7*M_PI)

inline double MagnetoTelluricPointOptions::toAbsoluteValue(double frequency, double apparentResistivity)
{
  return ::sqrt(apparentResistivity*(2*M_PI*frequency)/MAGNETIC_CONSTANT);
}

inline double MagnetoTelluricPointOptions::toApparentResistivity(double frequency, double absoluteValue)
{
  return absoluteValue*absoluteValue*MAGNETIC_CONSTANT/(2*M_PI*frequency);
}

} // namespace QGpCoreWave

#endif // MAGNETOTELLURICPOINTOPTIONS_H
