/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FUNCTIONSEARCH_H
#define FUNCTIONSEARCH_H

#include "QGpCoreMathDLLExport.h"
#include "AbstractFunction.h"
#include "Point2D.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT FunctionSearchMaximum
  {
  public:
    FunctionSearchMaximum(double value, const Point& pos) {
      _value=value;
      _pos=pos;
    }
    ~FunctionSearchMaximum() {}

    static bool greaterThanValue(const FunctionSearchMaximum *v1, const FunctionSearchMaximum *v2) {
      return v1->value()>v2->value();
    }
    static bool lessThanPos(const FunctionSearchMaximum *v1, const FunctionSearchMaximum *v2) {
      double dx=v1->pos().x()-v2->pos().x();
      if(fabs(dx)<fabs(v1->pos().x())*relativePrecision) {
        return v1->pos().y()-v2->pos().y()<0.0;
      }
      return dx<0.0;
    }
    static bool equalPos(const FunctionSearchMaximum *v1, const FunctionSearchMaximum *v2) {
      Point2D p=v1->pos();
      p-=v2->pos();
      return p.abs2()<v1->pos().abs2()*relativePrecision2;
    }

    double value() const {return _value;}
    const Point& pos() const {return _pos;}

    static double relativePrecision;
    static double relativePrecision2;
  private:
    double _value;
    Point _pos;
  };

  typedef QList<FunctionSearchMaximum *>::iterator FunctionMaximaIterator;

  class QGPCOREMATH_EXPORT FunctionSearch
  {
  public:
    FunctionSearch();
    virtual ~FunctionSearch();

    void setFunction(AbstractFunction * f);
    AbstractFunction * function() const {return _function;}
    AbstractFunction * takeFunction();

    double value(const Point& p) const {return _function->value(p);}
    double value(const Point& p, int index) const {return _function->value(p, index);}

    const Point& pos() const {return _pos;}
    virtual void globalMax(double absThres=0.0);
    virtual void localMax(int nMax, double absThres=0.0, double relThres=0.0);
    virtual void localMaxBest(int nMax, double absThres=0.0, double relThres=0.0) {
      localMax(nMax, absThres, relThres);
    }
    void unique();

    void clearMaxima();
    int maximaCount() const {return _maxima.count();}
    FunctionMaximaIterator begin() {return _maxima.begin();}
    FunctionMaximaIterator end() {return _maxima.end();}

    void setRelativePrecision(int axis, double p) {_precision[axis]=p; _relative[axis]=true;}
    void setAbsolutePrecision(int axis, double p) {_precision[axis]=p; _relative[axis]=false;}
    double precision(int axis) const {return _precision[axis];}
    bool relative(int axis) const {return _relative[axis];}

    void debugScan(const Point& min, const Point& max, int n);
  protected:
    static inline double lowLimit(int i, double * axis);
    static inline double highLimit(int i, double * axis, int nAxis);

    Point _pos;
    AbstractFunction * _function;
    bool _relative[3];
    double _precision[3];
    QList<FunctionSearchMaximum *> _maxima;
  };

  inline double FunctionSearch::lowLimit(int i, double * axis)
  {
    return i>0 ? axis[i-1] : axis[0];
  }

  inline double FunctionSearch::highLimit(int i, double * axis, int nAxis)
  {
    return i<nAxis-1 ? axis[i+1] : axis[nAxis-1];
  }

} // namespace QGpCoreMath

#endif // FUNCTION2SEARCH.H
