/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2012-11-27
**  Copyright: 2012-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Ellipse.h"

namespace QGpCoreMath{

  /*!
    \class Ellipse Ellipse.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Default constructor
  */
  Ellipse::Ellipse()
  {
    _majorRadius=0.0;
    _minorRadius=0.0;
  }

  /*!
    Constructor
  */
  Ellipse::Ellipse::Ellipse(const Point2D& c, double rmaj, double rmin, double orientation)
  {
    _center=c;
    _majorRadius=fabs(rmaj);
    _minorRadius=rmin>0.0 ? rmaj : _majorRadius;
    _orientation.setRadians(orientation);
  }

  /*!
    Copy constructor
  */
  Ellipse::Ellipse(const Ellipse& o)
  {
    _center=o._center;
    _majorRadius=o._majorRadius;
    _minorRadius=o._minorRadius;
    _orientation=o._orientation;
  }

  void Ellipse::setOrientation(double o)
  {
    _orientation.setRadians(o);
    _orientation.initDegrees();
  }

  /*!
    Returns the radii projected along X axis.
  */
  double Ellipse::xRadius() const
  {
    /* Parametric form of the ellipse:
       X(t)=Xc+a*cos(t)*cos(phi)-b*sin(t)*sin(phi)
       The xRadius is the maximum X
       X'(t)=-a*sin(t)*cos(phi)-b*cos(t)*sin(phi)=0
          tan(t)=-b*tan(phi)/a
    */
    if(_majorRadius>0.0) {
      double t=atan(-_minorRadius/_majorRadius*_orientation.tan());
      return fabs(_majorRadius*cos(t)*_orientation.cos()-_minorRadius*sin(t)*_orientation.sin());
    } else {
      return 0.0;
    }
  }

  /*!
    Returns the radii projected along Y axis.
  */
  double Ellipse::yRadius() const
  {
    /* Parametric form of the ellipse:
       Y(t)=Yc+a*cos(t)*sin(phi)+b*sin(t)*cos(phi)
       The yRadius is the maximum Y
       Y'(t)=-a*sin(t)*sin(phi)+b*cos(t)*cos(phi)=0
          tan(t)=b/(a*tan(phi))
    */
    if(_majorRadius>0.0) {
      if(_orientation.radians()!=0.0) {
        double t=atan(_minorRadius/(_majorRadius*_orientation.tan()));
        return fabs(_majorRadius*cos(t)*_orientation.sin()+_minorRadius*sin(t)*_orientation.cos());
      } else {
        return _minorRadius;
      }
    } else {
      return 0.0;
    }
  }

  QString Ellipse::toString() const
  {
    return QString("%1 %2 %3 %4").arg(_center.toString())
        .arg(_majorRadius).arg(_minorRadius).arg(_orientation.degrees());
  }

} // namespace QGpCoreMath