/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-07
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef CURVE_H
#define CURVE_H

#include <math.h>
#include <QGpCoreTools.h>

#include "Point2D.h"
#include "SmoothingParameters.h"
#include "CurvePointOptions.h"
#include "QGpCoreMathDLLExport.h"
#include "Circle.h"
#include "CurveSplitter.h"
#include "LeastSquare.h"
#include "SamplingParameters.h"

namespace QGpCoreMath {

  template <class pointType>
  class QGPCOREMATH_EXPORT Curve : private QVector<pointType>
  {
    TRANSLATIONS("Curve")
  public:
    Curve() {_sorted=false;}
    Curve(int n) : QVector<pointType>(n) {_sorted=false;}
    Curve(const Curve<pointType>& o) : QVector<pointType>(o) {_sorted=o._sorted;}
    Curve(const QVector<pointType>& o) : QVector<pointType>(o) {_sorted=false;}

    void sort();
    void checkSort();
    bool isSorted() const {return _sorted;}
    void unique();

    bool operator==(const Curve<pointType>& o) const {return QVector<pointType>::operator==(o);}
    bool operator<(const Curve<pointType>& o) const;
    int count() const {return QVector<pointType>::count();}
    bool isEmpty() const {return QVector<pointType>::isEmpty();}
    void resize(int n) {if(n>count()) {_sorted=false;} QVector<pointType>::resize(n);}
    int indexAfter(double valX) const;
    int indexOf(double valX, double relPrec) const;
    int indexOf(const pointType& p) const {return QVector<pointType>::indexOf(p);}
    void indexRange(double xMin, double xMax, int& indexMin, int& indexMax) const;
    pointType interpole(double x) const;
    bool isInsideRange(double x) const;

    const pointType& constAt(int index) const {return QVector<pointType>::at(index);}
    const pointType& constAt(int index) {return QVector<pointType>::at(index);}
    pointType& constXAt(int index) {return QVector<pointType>::operator[](index);}
    pointType& at(int index) {_sorted=false; return QVector<pointType>::operator[](index);}

    double x(int index) const {return constAt(index).x();}
    double firstX() const {return QVector<pointType>::first().x();}
    double lastX() const {return QVector<pointType>::last().x();}
    void setX(int index, double x) {at(index).setX(x);}
    inline double y(int index, const CurvePointOptions * pointOptions=nullptr) const;
    double firstY(const CurvePointOptions * pointOptions=nullptr) const {return QVector<pointType>::first().y(pointOptions);}
    double lastY(const CurvePointOptions * pointOptions=nullptr) const {return QVector<pointType>::last().y(pointOptions);}
    inline void setY(int index, double y, const CurvePointOptions * pointOptions=nullptr);
    bool isValid(int index) const {return constAt(index).isValid();}
    void setValid(int index, bool v) {constXAt(index).setValid(v);}

    void remove(int index) {QVector<pointType>::remove(index);}
    void insert(int index, const pointType& p) {_sorted=false; QVector<pointType>::insert(index, p);}
    void insert(const pointType& p);
    void clear() {_sorted=false; QVector<pointType>::clear();}
    void prepend(const pointType& p);
    void append(const pointType& p);
    void append(const Curve<pointType>& c);
    typedef pointType * iterator;
    typedef const pointType * const_iterator;
    const_iterator begin() const {return QVector<pointType>::begin();}
    const_iterator end() const {return QVector<pointType>::end();}
    iterator begin() {_sorted=false; return QVector<pointType>::begin();}
    iterator end() {_sorted=false; return QVector<pointType>::end();}

    bool sameSampling(const Curve<pointType>& o) const;

    void xLog();
    void xExp();
    void xMultiply(double factor);
    void xInverse();

    void ySetValue(double value, const CurvePointOptions * pointOptions=nullptr);
    void ySetMinimumValue(double value, const CurvePointOptions * pointOptions=nullptr);
    void ySetMaximumValue(double value, const CurvePointOptions * pointOptions=nullptr);
    void ySum(double value, const CurvePointOptions * pointOptions=nullptr);
    void ySum(const Curve<pointType>& o, const CurvePointOptions * pointOptions=nullptr);
    void yMultiply(double factor, const CurvePointOptions * pointOptions=nullptr);
    void yMultiply(const Curve<pointType>& o, const CurvePointOptions * pointOptions=nullptr);
    void yDivide(const Curve<pointType>& o, const CurvePointOptions * pointOptions=nullptr);
    void yMultiplyByX(double factor, const CurvePointOptions * pointOptions=nullptr);
    void yDivideByX(double factor, const CurvePointOptions * pointOptions=nullptr);
    void ySquare(const CurvePointOptions * pointOptions=nullptr);
    void ySqrt(const CurvePointOptions * pointOptions=nullptr);
    void yInverse(const CurvePointOptions * pointOptions=nullptr);
    void yLog(const CurvePointOptions * pointOptions=nullptr);
    void yExp(const CurvePointOptions * pointOptions=nullptr);
    void yAbs(const CurvePointOptions * pointOptions=nullptr);

    void line(const pointType& p1, const pointType& p2);
    void resample(int n, double min, double max, double valX, double valY,
                  SamplingOptions options=LinearScale,
                  const CurvePointOptions * pointOptions=nullptr);
    void resample(int n, double min, double max, SamplingOptions options=LinearScale);
    void resample(const Curve<pointType>& xModel);
    void resample(const QVector<double>& xModel);
    void cut(double min, double max, SamplingOptions options=LinearScale);
    void validate(double min, double max, bool value, SamplingOptions options=LinearScale);
    void smooth(const SamplingParameters& sampling, const SmoothingParameters& param,
                const CurvePointOptions * pointOptions=nullptr);
    pointType smooth(double x, const SmoothingParameters& param,
                     const CurvePointOptions * pointOptions=nullptr);
    void average(const Curve<pointType>& o);
    bool extrapolateFromStart(int leastSquareCount, bool positiveSlopeOnly,
                              SamplingOptions options=LinearScale, const CurvePointOptions * pointOptions=nullptr);
    bool extrapolateFromEnd(int leastSquareCount, bool positiveSlopeOnly,
                            SamplingOptions options=LinearScale, const CurvePointOptions * pointOptions=nullptr);

    bool leastSquare(double& a, double& b, const CurvePointOptions * pointOptions=nullptr) const;
    double azimuth(const CurvePointOptions * pointOptions=nullptr) const;
    QVector<double> project(const CurvePointOptions * pointOptions=nullptr) const;

    void swapXY(const CurvePointOptions * pointOptions=nullptr);
    Curve<pointType> derivative(const CurvePointOptions * pointOptions=nullptr) const;
    Curve<pointType> curvature(const CurvePointOptions * pointOptions=nullptr) const;
    Curve<pointType> angles(const CurvePointOptions * pointOptions=nullptr) const;
    QList<Curve<pointType>> split(double maxX, SamplingOptions options,
                                  double maxErr, int minCount,
                                  const CurvePointOptions * pointOptions=nullptr) const;

    void setValid(bool v);
    void removeInvalid();
    int minimumX(int startIndex=0) const;
    int maximumX(int startIndex=0) const;
    int minimumY(int startIndex=0, const CurvePointOptions * pointOptions=nullptr) const;
    int maximumY(int startIndex=0, const CurvePointOptions * pointOptions=nullptr) const;
    int localMaximumY(int startIndex, const CurvePointOptions * pointOptions=nullptr) const;
    int localMinimumY(int startIndex, const CurvePointOptions * pointOptions=nullptr) const;
    double minimumStep(SamplingOptions options);
    int closestMax(int index);

    QVector<double> xVector() const;
    Curve<Point2D> xyCurve() const;

    QString toString(int precision=6, char format='g') const;
    bool fromStream(QTextStream& s, QString * comments=nullptr);

    void operator+=(const Curve<pointType>& o);
    void operator*=(const Curve<pointType>& o);

    double sumX() const;
    double sumY(const CurvePointOptions * pointOptions=nullptr) const;
    double sumXY(const CurvePointOptions * pointOptions=nullptr) const;
    double sumX2() const;
    double sumY2(const CurvePointOptions * pointOptions=nullptr) const;
  protected:
    bool _sorted;
  private:
    void firstValid(int& startIndex) const;
    inline static bool lessThan(const pointType& p1, const pointType& p2);
  };

  template <class pointType>
  inline double Curve<pointType>::y(int index, const CurvePointOptions * pointOptions) const
  {
    return constAt(index).y(pointOptions);
  }

  template <class pointType>
  inline void Curve<pointType>::setY(int index, double y, const CurvePointOptions * pointOptions)
  {
    constXAt(index).setY(y, pointOptions);
  }

  template <class pointType>
  void Curve<pointType>::prepend(const pointType& p)
  {
    if(isEmpty()) {
      _sorted=true;
    } else if(p.x()>=QVector<pointType>::first().x()) {
      _sorted=false;
    }
    QVector<pointType>::prepend(p);
  }

  template <class pointType>
  void Curve<pointType>::append(const pointType& p)
  {
    if(isEmpty()) {
      _sorted=true;
    } else if(p.x()<=QVector<pointType>::last().x()) {
      _sorted=false;
    }
    QVector<pointType>::append(p);
  }

  template <class pointType>
  void Curve<pointType>::append(const Curve<pointType>& c)
  {
    if(isEmpty()) {
      _sorted=c._sorted;
    } else if(!c.isEmpty() && QVector<pointType>::last().x()>=c.QVector<pointType>::first().x()) {
      _sorted=false;
    }
    QVector<pointType>::operator+=(c);
  }

  template <class pointType>
  void Curve<pointType>::sort()
  {
    if(!isEmpty()) {
      std::sort(begin(), end(), lessThan);
      // In case of lot of double entries, this is function was quite long with a remove()
      Curve<pointType> f;
      f.reserve(f.count());
      f.append(constAt(0));
      for(int i=1; i<count(); i++) {
        if(x(i-1)!=x(i)) {
          f.append(constAt(i));
        }
      }
      f.squeeze();
      // This is quite fast with implicit sharing
      *this=f;
      _sorted=true;
    }
  }

  template <class pointType>
  void Curve<pointType>::checkSort()
  {
    int n=count();
    for(int i=1; i<n; i++) {
      if(x(i-1)>=x(i)) {
        _sorted=false;
        return;
      }
    }
    _sorted=true;
  }

  template <class pointType>
  bool Curve<pointType>::operator<(const Curve<pointType>& o) const
  {
    int n=count();
    if(o.count()<n) {
      n=o.count();
    }
    for(int i=0; i<n; i++) {
      if(constAt(i)<o.constAt(i)) {
        return true;
      } else if(constAt(i)>o.constAt(i)) {
        return false;
      }
    }
    return count()<o.count();
  }

  template <class pointType>
  void Curve<pointType>::unique()
  {
    ::QGpCoreTools::unique(*this);
  }

  template <class pointType>
  inline bool Curve<pointType>::lessThan(const pointType& p1, const pointType& p2)
  {
    return p1.x()<p2.x();
  }

  template <class pointType>
  int Curve<pointType>::indexAfter(double valX) const
  {
    ASSERT(_sorted);
    ASSERT(!isEmpty());
    if(valX<=firstX()) {
      return 0;
    }
    int lasti=count()-1;
    if(valX>x(lasti)) {
      return lasti+1;
    }
    int i=1;
    while(i<count()) {
      i=i << 1;
    }
    int step2=i >> 1;
    while(step2>0) {
      if(i>lasti) {
        i-=step2;
      } else if(valX <= x(i)) {
        if(valX>x(i-1)) break;
        i-=step2;
      } else {
        i+=step2;
      }
      step2=step2 >> 1;
    }
    return i;
  }

  template <class pointType>
  int Curve<pointType>::indexOf(double valX, double relPrec) const
  {
    ASSERT(_sorted);
    double valXLow, valXHigh;
    if(valX>0.0) {
      valXLow=valX*(1.0-relPrec);
      valXHigh=valX*(1.0+relPrec);
    } else {
      valXLow=valX*(1.0+relPrec);
      valXHigh=valX*(1.0-relPrec);
    }
    ASSERT(!isEmpty());
    if(valXHigh<firstX()) {
      return -1;
    } else if(valXLow<firstX()) {
      return 0;
    }
    int lasti=count()-1;
    if(valXLow>x(lasti)) {
      return -1;
    }
    int i=1;
    while(i<count()) {
      i=i << 1;
    }
    int step2=i >> 1;
    while(step2>0) {
      if(i>lasti) {
        i-=step2;
      } else if(valXHigh<x(i)) {
        i-=step2;
      } else if(valXLow<x(i)) {
        return i;
      } else {
        i+=step2;
      }
      step2=step2 >> 1;
    }
    if(valXLow<x(i) && valXHigh>x(i)) {
      return i;
    } else {
      return -1;
    }
  }

  template <class pointType>
  void Curve<pointType>::indexRange(double xMin, double xMax, int& indexMin, int& indexMax) const
  {
    ASSERT(_sorted);
    int i=indexAfter(xMin)-1;
    if(i>indexMin) {
      indexMin=i;
      if(indexMin<0) {
        indexMin=0;
      }
    }
    i=indexAfter(xMax);
    if(i<indexMax) {
      indexMax=i;
      if(indexMax>=count()) {
        indexMax=count()-1;
      }
    }

  }

  template <class pointType>
  bool Curve<pointType>::isInsideRange(double x) const
  {
    ASSERT(_sorted);
    if(count()<2) {
      if(count()==1) {
        return firstX()==x;
      } else {
        return false;
      }
    } else {
      return firstX()<=x && x<=lastX();
    }
  }

  template <class pointType>
  pointType Curve<pointType>::interpole(double x) const
  {
    ASSERT(_sorted);
    if(count()<2) {
      if(count()==1) {
        return constAt(0);
      } else {
        return pointType();
      }
    } else {
      int i=indexAfter(x);
      if(i==0) i++;                // Force interpolation if outside range
      else if(i==count()) i--;
      pointType p;
      p.interpole(x, constAt(i-1), constAt(i));
      return p;
    }
  }

  template <class pointType>
  void Curve<pointType>::insert(const pointType& p)
  {
    TRACE;
    if(count()<2) {
      if(isEmpty()) {
        QVector<pointType>::append(p);
      } else if(p.x()<firstX()) {
        QVector<pointType>::prepend(p);
      } else if(p.x()>firstX()) {
        QVector<pointType>::append(p);
      } else {
        QVector<pointType>::operator[](0)=p;
      }
      _sorted=true;
    } else {
      int i=indexAfter(p.x());
      if(i<count() && x(i)==p.x()) {
        (*this)[i]=p;
      } else {
        if(interpole(p.x()).y()!=p.y()) {
          insert(i, p);
        }
      }
    }
  }

  template <class pointType>
  void Curve<pointType>::resample(int n, double min, double max, double valX, double valY,
                                  SamplingOptions options, const CurvePointOptions * pointOptions)
  {
    TRACE;
    // Transforms X scale according to options (log and inv)
    if(options & InversedScale) {
      xInverse();
      valX=1.0/valX;
    }
    if(options & LogScale) {
      xLog();
      valX=::log(valX);
      if(_sorted) {
        min=::log(min);
        max=::log(max);
      }
    }
    if(_sorted) {
      if(min<QVector<pointType>::first().x())
        min=QVector<pointType>::first().x();
      if(max>QVector<pointType>::last().x())
        max=QVector<pointType>::last().x();
    }
    int currentN=count();

    // Calculate the total "distance" curve
    Curve<Point2D> distances(currentN);
    distances.setX(0, firstX());
    distances.setY(0, 0.0);
    for(int i=1; i<currentN; i++) {
      const pointType& p1=constAt(i-1);
      const pointType& p2=constAt(i);
      distances.setX(i, p2.x());
      double deltaX=(p2.x()-p1.x())/valX;
      double deltaY=(p2.y(pointOptions)-p1.y(pointOptions))/valY;
      distances.setY(i, distances.y(i-1)+::sqrt(deltaX*deltaX+deltaY*deltaY));
    }
    distances.sort();

    // Calculate the distance of min and max, and constant step
    double distMin, distMax;
    if(_sorted) {
      Point2D p;
      int i;
      i=distances.indexAfter(min);
      p.interpole(min, distances.constAt(i-1), distances.constAt(i));
      distMin=p.y();
      i=distances.indexAfter(max);
      p.interpole(max, distances.constAt(i-1), distances.constAt(i));
      distMax=p.y();
    } else {
      distMin=0;
      distMax=distances.y(currentN-1);
    }
    double cstStep=(distMax-distMin)/(currentN-1);

    // Map x as a function of distance, function monotoneous, then no need to setFunction()
    distances.swapXY();

    // Intepolate x for constant step distance
    QVector<double> x(n);
    double dist=distMin;
    Point2D p;
    for(int i=0; i<n; i++, dist+=cstStep) {
      int iDist=distances.indexAfter(dist );
      p.interpole(dist, distances.constAt(iDist-1), distances.constAt(iDist));
      x[i]=p.y();
    }
    resample(x);

    // Re-Transforms axis according options (log and inv)
    if(options & LogScale) {
      xExp();
    }
    if(options & InversedScale) {
      xInverse();
    }
  }

  template <class pointType>
  void Curve<pointType>::resample(int n, double min, double max, SamplingOptions options)
  {
    TRACE;
    ASSERT(_sorted);
    // Transforms X scale according to options (log and inv)
    if(options & InversedScale) {
      xInverse();
      double tmp=min;
      min=1.0/max;
      max=1.0/tmp;
    }
    if(options & LogScale) {
      xLog();
      min=::log(min);
      max=::log(max);
    }
    // Calculate the constant step
    double cstStep=(max-min)/(n-1);
    QVector<double> x(n);
    for(int i=0; i<n; i++) {
      x[i]=min+(double)i*cstStep;
    }
    resample(x);

    // Re-Transforms axis according options (log and inv)
    if(options & LogScale) {
      xExp();
    }
    if(options & InversedScale) {
      xInverse();
    }
  }

  template <class pointType>
  void Curve<pointType>::resample(const Curve<pointType>& xModel)
  {
    TRACE;
    ASSERT(_sorted);
    QVector<double> x(xModel.count());
    for(int i=0; i<xModel.count(); i++) {
      x[i]=xModel.x(i);
    }
    resample(x);
  }

  template <class pointType>
  void Curve<pointType>::resample(const QVector<double>& xModel)
  {
    TRACE;
    ASSERT(_sorted);
    int nCurve=count();
    int nModel=xModel.count();
    if(nCurve<2 || nModel<2) return;
    Curve<pointType> interpolated(nModel);
    // Insert values inside xModel
    int iCurve=1;
    int lastCurveIndex=nCurve-1;
    for(int i=0; i<nModel; i++) {
      double x=xModel.at(i);
      while(iCurve<lastCurveIndex &&
            Curve<pointType>::x(iCurve)<x) {
        iCurve++;
      }
      const pointType& p1=constAt(iCurve-1);
      const pointType& p2=constAt(iCurve);
      interpolated[i].interpole(x, p1, p2);
      if(p1.isValid()) {
        if(p2.isValid()) {
          // Both are valid, avoid extrapolation
          if(x<p1.x()*(1+(p1.x()>0 ? -1e-15 : 1e-15)) ||
             x>p2.x()*(1+(p2.x()>0 ? 1e-15 : -1e-15))) {
            interpolated[i].setValid(false);
          }
        } else {
          // First is valid, second is not valid
          if(x>p1.x()*(1+(p1.x()>0 ? 1e-15 : -1e-15))) {
            interpolated[i].setValid(false);
          }
        }
      } else {
        if(!p2.isValid()) {
          interpolated[i].setValid(false);
        } else {
          // First is not valid, second is valid
          if(x<p2.x()*(1+(p2.x()>0 ? -1e-15 : 1e-15))) {
            interpolated[i].setValid(false);
          }
        }
      }
    }
    // Insert all values outside xModel
    if(nModel>1) {
      double min=xModel.at(0)*(1+(xModel.at(0)>0 ? -1e-15 : 1e-15));
      double max=xModel.at(nModel-1)*(1+(xModel.at(nModel-1)>0 ? 1e-15 : -1e-15));
      for(int i=0;i<nCurve;i++) {
        const pointType& p=constAt(i);
        if(p.x()<min || max<p.x()) {
          interpolated.append(p);
        }
      }
    }
    // If xModel is not sorted, a sort may be necessary
    interpolated.sort();
    *this=interpolated;
  }

  template <class pointType>
  void Curve<pointType>::cut(double min, double max, SamplingOptions options)
  {
    TRACE;
    ASSERT(_sorted);
    // Assert that count()>1 is made by indexAfter()
    // Transforms X scale according to options (log and inv)
    if(options & InversedScale) {
      xInverse();
    }
    if(min<QVector<pointType>::first().x())
      min=QVector<pointType>::first().x();
    if(max>QVector<pointType>::last().x())
      max=QVector<pointType>::last().x();
    // Find the first point
    int ix=indexAfter(min);
    if(ix>0) {
      if(fabs(min-x(ix-1)) < 1e-15) {
        ix--;
      } else if((options & Interpole) && fabs(min-x(ix))>1e-15) {
        ix--;
        QVector<pointType>::operator[](ix).interpole(min, constAt(ix), constAt(ix+1));
      }
    }
    // Remove everything before
    for(int i=0; i<ix; i++) {
      remove(0);
    }
    // Find the last point
    ix=indexAfter(max);
    if(ix < count()) {
      if(fabs(max-x(ix))<1e-15) {
        ix++;
      } else if((options & Interpole) && fabs(max-x(ix-1))>1e-15) {
        QVector<pointType>::operator[](ix).interpole(max, constAt(ix-1), constAt(ix));
        ix++;
      }
    }
    // Remove everything after
    for(int i=count()-1; i>=ix; i--) {
      remove(i);
    }
    // Re-Transforms axis according options (log and inv)
    if(options & InversedScale) {
      xInverse();
    }
  }

  template <class pointType>
  void Curve<pointType>::validate(double min, double max, bool value, SamplingOptions options)
  {
    TRACE;
    ASSERT(_sorted);
    // Assert that count()>1 is made by indexAfter()
    // Transforms X scale according to options (log and inv)
    if(options & InversedScale) {
      xInverse();
    }
    if(min<QVector<pointType>::first().x())
      min=QVector<pointType>::first().x();
    if(max>QVector<pointType>::last().x())
      max=QVector<pointType>::last().x();
    // Find the first and last points
    int ixs=indexAfter(min);
    int ixe=indexAfter(max);
    if(ixs>0) { // convenient rounding errors
      if(fabs(min-x(ixs-1))<1e-15) {
        ixs--;
      }
    }
    if(ixe>0) { // convenient rounding errors
      if(fabs(min-x(ixe-1))<1e-15) {
        ixe--;
      }
    }
    if(ixe==count()) {
      ixe--;
    }
    for(int i=ixs; i<=ixe; i++) {
      QVector<pointType>::operator[](i).setValid(value);
    }
    // Re-Transforms axis according options (log and inv)
    if(options & InversedScale) {
      xInverse();
    }
  }

  template <class pointType>
  void Curve<pointType>::smooth(const SamplingParameters& sampling, const SmoothingParameters& param,
                                const CurvePointOptions * pointOption)
  {
    TRACE;
    ASSERT(_sorted);
    int n=sampling.count();
    Curve<pointType> interpolated(n);
    for(int i=0; i<n; i++) {
      interpolated[i]=smooth(sampling.value(i), param, pointOption);
    }
    interpolated.sort();
    *this=interpolated;
  }

  template <class pointType>
  pointType Curve<pointType>::smooth(double x, const SmoothingParameters& param,
                                     const CurvePointOptions * pointOption)
  {
    TRACE;
    pointType sum;
    if(count()<2) {
      if(count()==1) {
        return constAt(0);
      } else {
        return sum;
      }
    }
    double min=param.minimum(x);
    double max=param.maximum(x);
    int i0=indexAfter(min);
    int n=indexAfter(max);
    if(i0>=n) {
      if(i0==count()) {
        i0--;
      }
      if(i0==0) {
        i0++;
      }
      sum.interpole(x, constAt(i0-1), constAt(i0));
      return sum;
    }
    switch(param.method()) {
    case SmoothingParameters::Function: {
        double sumWeight=0.0;
        switch(param.scaleType()) {
        case SamplingParameters::Linear:
          for(int i=i0; i<n; i++) {
            pointType p=constAt(i);
            double weight=WindowFunction::value(p.x(), x, min, max, param.windowFunction());
            sumWeight+=weight;
            p*=weight;
            sum+=p;
          }
          sum/=sumWeight;
          break;
        case SamplingParameters::Log:
          for(int i=i0; i<n; i++) {
            pointType p=constAt(i);
            p.yLog();
            double weight=WindowFunction::value(p.x(), x, min, max, param.windowFunction());
            sumWeight+=weight;
            p*=weight;
            sum+=p;
          }
          sum/=sumWeight;
          sum.yExp();
          break;
        case SamplingParameters::Inversed:
          for(int i=i0; i<n; i++) {
            pointType p=constAt(i);
            p.yInverse();
            double weight=WindowFunction::value(p.x(), x, min, max, param.windowFunction());
            sumWeight+=weight;
            p*=weight;
            sum+=p;
          }
          sum/=sumWeight;
          sum.yInverse();
          break;
        }
      }
      break;
    case SmoothingParameters::SavitzkyGolay: {
        LeastSquare s(n-i0, param.order());
        s.setScaleType(param.scaleType());
        for(int i=i0; i<n; i++) {
          const pointType& p=constAt(i);
          s.setData(i-i0, p.x(), p.y(pointOption));
        }
        s.invert();
        sum.setY(s.value(x), pointOption);
      }
      break;
    }
    sum.setX(x);
    return sum;
  }

  template <class pointType>
  void Curve<pointType>::average(const Curve<pointType>& o)
  {
    TRACE;
    if(o.count()<2) return;
    if(isEmpty()) {
      *this=o;
      return;
    }
    // Set the same sampling for both curves
    QVector<double> x=xVector();
    x+=o.xVector();
    std::sort(x.begin(), x.end());
    ::QGpCoreTools::unique(x);
    if(!isSorted()) {
      sort();
    }
    resample(x);
    Curve<pointType> oResamp=o;
    if(!oResamp.isSorted()) {
      oResamp.sort();
    }
    oResamp.resample(x);
    // Take the average of the two curves
    for(int i=x.count()-1; i>=0; i--) {
      QVector<pointType>::operator[](i).average(oResamp.at(i));
    }
    sort();
  }

  template <class pointType>
  void Curve<pointType>::setValid(bool v)
  {
    TRACE;
    iterator it;
    for(it=begin(); it!=end(); ++it) {
      it->setValid(v);
    }
  }

  template <class pointType>
  void Curve<pointType>::removeInvalid()
  {
    TRACE;
    for(int i=count()-1; i>=0; i--) {
      if(!isValid(i)) {
        remove(i);
      }
    }
  }

  template <class pointType>
  void Curve<pointType>::swapXY(const CurvePointOptions * pointOptions)
  {
    TRACE;
    iterator it;
    for(it=QVector<pointType>::begin(); it!=QVector<pointType>::end(); ++it) {
      double tmp=it->x();
      it->setX(it->y(pointOptions));
      it->setY(tmp, pointOptions);
    }
    _sorted=false;
  }

  template <class pointType>
  Curve<pointType> Curve<pointType>::derivative(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    ASSERT(_sorted);
    int n=count()-1;
    ASSERT(n>1);
    Curve<pointType> d(n-1);
    for(int i=1; i<n; i++) {
      double v=(y(i, pointOptions)-y(i-1, pointOptions))
                 /(x(i)-x(i-1));
      v+=(y(i+1, pointOptions)-y(i, pointOptions))
        /(x(i+1)-x(i));
      v*=0.5;
      pointType& p=d[i-1];
      p.setX(x(i));
      p.setY(v, pointOptions);
    }
    return d;
  }

  template <class pointType>
  Curve<pointType> Curve<pointType>::curvature(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    ASSERT(_sorted);
    int n=count()-1;
    ASSERT(n>1);
    Curve<pointType> d(n-1);
    for(int i=1; i<n; i++) {
      Circle c(constAt(i-1).point2D(pointOptions),
               constAt(i).point2D(pointOptions),
               constAt(i+1).point2D(pointOptions));
      pointType& p=d[i-1];
      p.setX(x(i));
      p.setY(c.radius(), pointOptions);
    }
    return d;
  }

  template <class pointType>
  Curve<pointType> Curve<pointType>::angles(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    ASSERT(_sorted);
    int n=count();
    ASSERT(n>1);
    Curve<pointType> d(n-1);
    for(int i=1; i<n; i++) {
      const pointType& p1=constAt(i-1);
      const pointType& p2=constAt(i);
      pointType& a=d[i-1];
      a.setX(p1.x());
      a.setY(p1.point2D(pointOptions).azimuthTo(p2.point2D(pointOptions)), pointOptions);
    }
    return d;
  }

  template <class pointType>
  void Curve<pointType>::xMultiply(double factor)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i); // Not really const but sort afterwards
      p.setX(p.x()*factor);
    }
    if(factor<0.0 && _sorted) {
      sort();
    }
  }

  template <class pointType>
  void Curve<pointType>::xInverse()
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i); // Not really const but sort afterwards
      p.setX(1.0/p.x());
    }
    if(_sorted) {
      sort();
    }
  }

  template <class pointType>
  void Curve<pointType>::xExp()
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i); // Not really const but exp does not alter order
      p.setX(exp(p.x()));
    }
  }

  template <class pointType>
  void Curve<pointType>::xLog()
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i); // Not really const but log does not alter order
      if(p.x()>0.0) {
        p.setX(::log(p.x()));
      }
    }
  }

  template <class pointType>
  void Curve<pointType>::ySetValue(double v, const CurvePointOptions * pointOptions)
  {
    int n=count();
    for(int i=0; i<n; i++) {
      setY(i, v, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::ySetMinimumValue(double v, const CurvePointOptions * pointOptions)
  {
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      if(p.y(pointOptions)<v) {
        p.setY(v, pointOptions);
      }
    }
  }

  template <class pointType>
  void Curve<pointType>::ySetMaximumValue(double v, const CurvePointOptions * pointOptions)
  {
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      if(p.y(pointOptions)>v) {
        p.setY(v, pointOptions);
      }
    }
  }

  template <class pointType>
  void Curve<pointType>::ySum(double value, const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)+value, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::ySum(const Curve<pointType>& o, const CurvePointOptions * pointOptions)
  {
    if(isEmpty()) {
      *this=o;
      return;
    }
    int n=count();
    if(n!=o.count()) {
      App::log(tr("Curve::ySum(): uncompatible sampling (count=%1 & %2).\n").arg(n).arg(o.count()) );
      return;
    }
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)+o.y(i, pointOptions), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yMultiply(double factor, const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)*factor, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yMultiply(const Curve<pointType>& o, const CurvePointOptions * pointOptions)
  {
    if(isEmpty()) {
      *this=o;
      return;
    }
    int n=count();
    if(n!=o.count()) {
      App::log(tr("Curve::yMultiply(): uncompatible sampling.\n") );
      return;
    }
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)*o.y(i, pointOptions), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yDivide(const Curve<pointType>& o, const CurvePointOptions * pointOptions)
  {
    if(isEmpty()) {
      *this=o;
      return;
    }
    int n=count();
    if(n!=o.count()) {
      App::log(tr("Curve::yDivide(): uncompatible sampling.\n") );
      return;
    }
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)/o.y(i, pointOptions), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yMultiplyByX(double factor, const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)*p.x()*factor, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yDivideByX(double factor, const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(p.y(pointOptions)/(p.x()*factor), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::ySquare(const CurvePointOptions * pointOptions)
  {
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      double v=p.y(pointOptions);
      p.setY(v*v, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::ySqrt(const CurvePointOptions * pointOptions)
  {
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      double v=p.y(pointOptions);
      if(v>=0.0) p.setY(::sqrt(v), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yInverse(const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      double v=p.y(pointOptions);
      if(v!=0.0) p.setY(1.0/v, pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yLog(const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      double v=p.y(pointOptions);
      if(v>0.0) p.setY(::log(v), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yExp(const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(exp(p.y(pointOptions)), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::yAbs(const CurvePointOptions * pointOptions)
  {
    TRACE;
    int n=count();
    for(int i=0; i<n; i++) {
      pointType& p=constXAt(i);
      p.setY(fabs(p.y(pointOptions)), pointOptions);
    }
  }

  template <class pointType>
  void Curve<pointType>::line(const pointType& p1, const pointType& p2)
  {
    TRACE;
    resize(2);
    if(p1.x()<p2.x()) {
      (*this)[0]=p1;
      (*this)[1]=p2;
    } else {
      (*this)[0]=p2;
      (*this)[1]=p1;
    }
    _sorted=true;
  }

  template <class pointType>
  QVector<double> Curve<pointType>::xVector() const
  {
    int n=count();
    QVector<double> x(n);
    for(int i=0; i<n; i++) {
      x[i]=Curve<pointType>::x(i);
    }
    return x;
  }

  template <class pointType>
  Curve<Point2D> Curve<pointType>::xyCurve() const
  {
    int n=count();
    Curve<Point2D> c(n);
    for(int i=0; i<n; i++) {
      c.at(i)=Point2D(x(i), y(i));
    }
    return c;
  }

  template <class pointType>
  void Curve<pointType>::firstValid(int& startIndex) const
  {
    int n=count();
    if(startIndex<0) {
      startIndex=0;
    } else if(startIndex>=n) {
      startIndex=n-1;
    }
    while(startIndex<n && !isValid(startIndex)) {
      startIndex++;
    }
  }

  template <class pointType>
  int Curve<pointType>::minimumX(int startIndex) const
  {
    int n=count();
    if(n<=0  || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex<n) {
      const pointType * p=&constAt(startIndex);
      int iMin=startIndex;
      for(int i=startIndex+1; i<n; i++) {
        const pointType& pi=constAt(i);
        if(pi.isValid() && pi.x()<p->x()) {
          p=&pi;
          iMin=i;
        }
      }
      return iMin;
    } else {
      return -1;
    }
  }

  template <class pointType>
  int Curve<pointType>::maximumX(int startIndex) const
  {
    int n=count();
    if(n<=0 || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex<n) {
      const pointType * p=&constAt(startIndex);
      int iMax=startIndex;
      for(int i=startIndex+1; i<n; i++) {
        const pointType& pi=constAt(i);
        if(pi.isValid() && pi.x()>p->x()) {
          p=&pi;
          iMax=i;
        }
      }
      return iMax;
    } else {
      return -1;
    }
  }

  template <class pointType>
  int Curve<pointType>::minimumY(int startIndex, const CurvePointOptions * pointOptions) const
  {
    int n=count();
    if(n<=0 || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex<n) {
      const pointType * p=&constAt(startIndex);
      int iMin=startIndex;
      for(int i=startIndex+1; i<n; i++) {
        const pointType& pi=constAt(i);
        if(pi.isValid() && pi.y(pointOptions)<p->y(pointOptions)) {
          p=&pi;
          iMin=i;
        }
      }
      return iMin;
    } else {
      return -1;
    }
  }

  template <class pointType>
  int Curve<pointType>::maximumY(int startIndex, const CurvePointOptions * pointOptions) const
  {
    int n=count();
    if(n<=0 || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex<n) {
      const pointType * p=&constAt(startIndex);
      int iMax=startIndex;
      for(int i=startIndex+1; i<n; i++) {
        const pointType& pi=constAt(i);
        if(pi.isValid() && pi.y(pointOptions)>p->y(pointOptions)) {
          p=&pi;
          iMax=i;
        }
      }
      return iMax;
    } else {
      return -1;
    }
  }

  template <class pointType>
  int Curve<pointType>::localMaximumY(int startIndex, const CurvePointOptions * pointOptions) const
  {
    int n=count();
    if(n<=0 || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex==n) {
      return -1;
    }
    const pointType * p0=&constAt(startIndex);
    firstValid(startIndex);
    if(startIndex==n) {
      return -1;
    }
    const pointType * p1=&constAt(startIndex);
    for(int i=startIndex; i<n; i++) {
      const pointType * p2=&constAt(i);
      if(p2->isValid()) {
        if(p1->y(pointOptions)>p0->y(pointOptions) &&
           p1->y(pointOptions)>p2->y(pointOptions)) {
          return i;
        }
      }
      p0=p1;
      p1=p2;
    }
    return -1;
  }

  template <class pointType>
  int Curve<pointType>::localMinimumY(int startIndex, const CurvePointOptions * pointOptions) const
  {
    int n=count();
    if(n<=0 || startIndex>=n) {
      return -1;
    }
    firstValid(startIndex);
    if(startIndex==n) {
      return -1;
    }
    const pointType * p0=&constAt(startIndex);
    firstValid(startIndex);
    if(startIndex==n) {
      return -1;
    }
    const pointType * p1=&constAt(startIndex);
    for(int i=startIndex; i<n; i++) {
      const pointType * p2=&constAt(i);
      if(p2->isValid()) {
        if(p1->y(pointOptions)<p0->y(pointOptions) &&
           p1->y(pointOptions)<p2->y(pointOptions)) {
          return i;
        }
      }
      p0=p1;
      p1=p2;
    }
    return -1;
  }

  template <class pointType>
  int Curve<pointType>::closestMax(int index)
  {
    TRACE;
    int n=count();
    int di;

    if(index==0) {
      di=1;
      index=1;
    }
    else if(index>=n) {
      index=n;
      di=-1;
    } else if(y(index)>y(index-1)) di=1;
    else if(y(index)<y(index-1)) di=-1;
    else return index;

    if(di>0) {
      while(index<n) {
        if(y(index)<y(index-1)) {
          index--;
          break;
        }
        index+=di;
      }
      if(index==n) index--;
    } else {
      while(index>0) {
        if(y(index)>y(index-1)) break;
        index+=di;
      }
    }
    return index;
  }

  template <class pointType>
  QString Curve<pointType>::toString(int precision, char format) const
  {
    TRACE;
    QString s;
    for(const_iterator it=begin(); it!=end(); ++it) {
      s+=it->toString(precision, format);
      s+="\n";
    }
    return s;
  }

  /*!
    Sum of all x (used in least square method)
  */
  template <class pointType>
  double Curve<pointType>::sumX() const
  {
    TRACE;
    double sum=0.0;
    const_iterator it;
    for(it=begin();it!=end();++it) sum+=it->x();
    return sum;
  }

  /*!
    Sum of all y (used in least square method)
  */
  template <class pointType>
  double Curve<pointType>::sumY(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    double sum=0.0;
    const_iterator it;
    for(it=begin();it!=end();++it) sum+=it->y(pointOptions);
    return sum;
  }

  /*!
    Sum of all x*y (used in least square method)
  */
  template <class pointType>
  double Curve<pointType>::sumXY(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    double sum=0.0;
    const_iterator it;
    for(it=begin();it!=end();++it) sum+=it->x()*it->y(pointOptions);
    return sum;
  }

  /*!
    Sum of all x*x (used in least square method)
  */
  template <class pointType>
  double Curve<pointType>::sumX2() const
  {
    TRACE;
    double sum=0.0;
    const_iterator it;
    for(it=begin();it!=end();++it) sum+=it->x()*it->x();
    return sum;
  }

  /*!
    Sum of all y*y (used in least square method)
  */
  template <class pointType>
  double Curve<pointType>::sumY2(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    double sum=0.0;
    const_iterator it;
    for(it=begin();it!=end();++it) sum+=it->y()*it->y(pointOptions);
    return sum;
  }

  /*!
    Performs a least square regression on all points in the horizontal plane.

    Returns true if there is a sufficient number of points in curve (minimum=2) and
    if \a a is not inf.
    Returns coefficient \a a and \a b from expression y=a*x+b passed as reference in arguments.
  */
  template <class pointType>
  bool Curve<pointType>::leastSquare(double& a, double& b, const CurvePointOptions * pointOptions) const
  {
    TRACE;
    int n=count();
    if(n>1) {
      double invn=1.0/(double)n;
      double sx=sumX();
      double sy=sumY(pointOptions);
      double sxy=sumXY(pointOptions);
      double sx2=sumX2();
      double denom=sx2-invn*sx*sx;
      if(denom!=0.0) {
        a=(sxy-invn*sy*sx)/denom;
        b=(invn*sy*sx2-invn*sxy*sx)/denom;
        return true;
      } else {
        a=0.0;
        b=sx*invn;
      }
    }
    return false;
  }

  /*!
    Performs a least square regression on all points in the horizontal plane.

    Returns the azimuth of the regression line (mathematical sense).
  */
  template <class pointType>
  double Curve<pointType>::azimuth(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    int n=count();
    if(n>1) {
      double a, b;
      if(leastSquare(a, b, pointOptions)) {
        return atan(a);
      } else {
        return 0.5*M_PI;
      }
    }
    return 0.0;
  }

  /*!
    Performs a least square regression on all points in the horizontal plane.

    Returns the distances along the profile
  */
  template <class pointType>
  QVector<double> Curve<pointType>::project(const CurvePointOptions * pointOptions) const
  {
    TRACE;
    int n=count();
    if(!n) return QVector<double>();
    QVector<double> projections(n);
    if(n>1) {
      double a, b;
      double min=std::numeric_limits<double>::infinity();
      if(leastSquare(a, b, pointOptions)) {
        for(int i=0; i<n; i++) {
          const pointType& p=constAt(i);
          double denom=a*a+1.0;
          double cp=-p.x()-a*p.y(pointOptions);
          double xp=(-cp-a*b)/denom;
          double yp=(-a*cp+b)/denom-b;
          double d=::sqrt(xp*xp+yp*yp);
          projections[i]=(xp>=0)? d : -d;
          if(projections[i]<min) min=projections[i];
        }
      } else {
        for(int i=0; i<n; i++) {
          const pointType& p=constAt(i);
          projections[i]=p.y(pointOptions);
          if(projections[i]<min) min=projections[i];
        }
      }
      // set min to abcsisse=0
      for(int i=0; i<n;i++) projections[i]-=min;
    } else {
      projections[0]=0;
    }
    return projections;
  }

  template <class pointType>
  bool Curve<pointType>::sameSampling(const Curve<pointType>& o) const
  {
    TRACE;
    int n=count();
    if(n!=o.count()) {
      return false;
    }
    for(int i=0; i<n; i++) {
      if(x(i)!=o.x(i)) {
        return false;
      }
    }
    return true;
  }

  template <class pointType>
  double Curve<pointType>::minimumStep(SamplingOptions options)
  {
    TRACE;
    ASSERT(_sorted);
    int n=count();
    double s, min=std::numeric_limits<double>::infinity();
    if(options & LogScale) {
      for(int i=1; i<n; i++) {
        s=x(i)/x(i-1);
        if(s<min) {
          min=s;
        }
      }
    } else {
      for(int i=1; i<n; i++) {
        s=x(i)-x(i-1);
        if(s<min) {
          min=s;
        }
      }
    }
    return min;
  }

  template <class pointType>
  QList<Curve<pointType>> Curve<pointType>::split(double maxX,
                                                  SamplingOptions options,
                                                  double maxErr,
                                                  int minCount,
                                                  const CurvePointOptions * pointOptions) const
  {
    TRACE;
    int n=count();
    CurveSplitter splitter;
    if(options & LogScale) {
      double invMaxX=1.0/maxX;
      if(maxX<invMaxX) {
        qSwap(maxX, invMaxX);
      }
      for(int i=0; i<n; i++) {
        if(isValid(i)) {
          double x0=x(i);
          double y0=y(i, pointOptions);
          for(int j=0; j<n; j++) {
            if(i!=j && isValid(j)) {
              double dx=x(j)/x(i);
              if((invMaxX<dx && dx<1.0-1e-12) || // Never add a pair at the same x value
                 (dx>1.0+1e-12 && dx<maxX)) {
                splitter.addPair(i, j, x0, y0, ::log(dx), y(j, pointOptions));
              }
            }
          }
        }
      }
    } else {
      for(int i=0; i<n; i++) {
        if(isValid(i)) {
          double x0=x(i);
          double y0=y(i, pointOptions);
          for(int j=0; j<n; j++) {
            double dx=x(j)-x(i);
            if(i!=j && isValid(j) && fabs(dx)<maxX) {
              if(fabs(dx)>1e-12) { // Never add a pair at the same x value
                splitter.addPair(i, j, x0, y0, dx, y(j, pointOptions));
             }
            }
          }
        }
      }
    }
    splitter.clusterize(maxErr);
    splitter.validSegments(maxErr, xyCurve());
    splitter.continuousSegments(xVector());
    int nc=splitter.clusterCount();
    QList<Curve<pointType>> list;
    for(int ic=0; ic<nc; ic++) {
      QVector<int> indexList=splitter.pointIndex(ic);
      int ni=indexList.count();
      if(ni>minCount) {
        Curve<pointType> c(ni);
        for(int ii=0; ii<ni; ii++) {
          c.at(ii)=constAt(indexList.at(ii));
        }
        c.sort();
        if(c.count()>minCount) { // If several samples have the same x, sort may reduce the number of samples
          list.append(c);
        }
      }
    }
    return list;
  }

  template <class pointType>
  bool Curve<pointType>::extrapolateFromStart(int leastSquareCount,
                                              bool positiveSlopeOnly,
                                              SamplingOptions options,
                                              const CurvePointOptions * pointOptions)
  {
    TRACE;
    ASSERT(_sorted);
    int n=count();
    double step=minimumStep(options);
    double a, b;

    Curve<Point2D> curveBegin;
    for(int i=0; i<n; i++) {
      const pointType& p=constAt(i);
      curveBegin.append(Point2D(p.x(), p.y(pointOptions)));
      if(curveBegin.count()==leastSquareCount) {
        break;
      }
    }
    if(curveBegin.count()<leastSquareCount) {
      return false;
    }

    // first and last for a non-const function mark this curve as not sorted
    bool extrapolated=false;
    double x=firstX();
    if(options & LogScale) {
      curveBegin.xLog();
      curveBegin.leastSquare(a, b, pointOptions);
      if(!positiveSlopeOnly || a>0.0) {
        for(int i=leastSquareCount-1; i>=0; i--) {
          pointType p;
          x/=step;
          p.setX(x);
          p.setY(a*log(x)+b, pointOptions);
          prepend(p);
        }
        extrapolated=true;
      }
    } else {
      curveBegin.leastSquare(a, b, pointOptions);
      if(!positiveSlopeOnly || a>0.0) {
        for(int i=leastSquareCount-1; i>=0; i--) {
          pointType p;
          x-=step;
          p.setX(x);
          p.setY(a*x+b, pointOptions);
          prepend(p);
        }
        extrapolated=true;
      }
    }

    // Above operations did not change the sort
    _sorted=true;
    return extrapolated;
  }

  template <class pointType>
  bool Curve<pointType>::extrapolateFromEnd(int leastSquareCount,
                                            bool positiveSlopeOnly,
                                            SamplingOptions options,
                                            const CurvePointOptions * pointOptions)
  {
    TRACE;
    ASSERT(_sorted);
    int n=count();
    double step=minimumStep(options);
    double a, b;

    Curve<Point2D> curveEnd;
    for(int i=n-1; i>=0; i--) {
      const pointType& p=constAt(i);
      curveEnd.append(Point2D(p.x(), p.y(pointOptions)));
      if(curveEnd.count()==leastSquareCount) {
        break;
      }
    }
    if(curveEnd.count()<leastSquareCount) {
      return false;
    }

    // first and last for a non-const function mark this curve as not sorted
    bool extrapolated=false;
    double x=Curve<pointType>::x(count()-1);
    if(options & LogScale) {
      curveEnd.xLog();
      curveEnd.leastSquare(a, b, pointOptions);
      if(!positiveSlopeOnly || a>0.0) {
        for(int i=leastSquareCount-1; i>=0; i--) {
          pointType p;
          x*=step;
          p.setX(x);
          p.setY(a*log(x)+b, pointOptions);
          append(p);
        }
        extrapolated=true;
      }
    } else {
      curveEnd.leastSquare(a, b, pointOptions);
      if(!positiveSlopeOnly || a>0.0) {
        for(int i=leastSquareCount-1; i>=0; i--) {
          pointType p;
          x+=step;
          p.setX(x);
          p.setY(a*x+b, pointOptions);
          append(p);
        }
        extrapolated=true;
      }
    }
    // Above operations did not change the sort
    _sorted=true;
    return extrapolated;
  }

  template <class pointType>
  bool Curve<pointType>::fromStream(QTextStream& s, QString * comments)
  {
    TRACE;
    pointType p;
    QString buf;
    clear();
    while(!s.atEnd()) {
      buf=s.readLine();
      if(buf.isEmpty() || buf[0]=='\n' || buf[0]=='#') {
        if(isEmpty()) {
          if(comments) {
            (*comments)+=buf+"\n";
          }
        } else {
          return true;
        }
      } else {
        p.fromString(buf);
        append(p);
      }
    }
    return false;
  }

} // namespace QGpCoreMath

#endif // CURVE_H
