/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-24
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef COLORMAP_H
#define COLORMAP_H

#include <QGpCoreTools.h>

#include "QGpCoreMathDLLExport.h"
#include "AbstractColorPalette.h"
#include "ColorMapData.h"

namespace QGpCoreMath {

class QGPCOREMATH_EXPORT ColorMap : public XMLClass, public AbstractColorPalette
{
  TRANSLATIONS( "ColorMap" )
public:
  ColorMap() {_d=new ColorMapData;}

  virtual const QString& xml_tagName() const {return xmlColorMapTag;}
  virtual bool xml_isValidTagName(const QString& t) const;
  static const QString xmlColorMapTag;

  bool operator==(const ColorMap& o) const {return _d->operator==(*o._d);}
  bool operator!=(const ColorMap& o) const {return _d->operator!=(*o._d);}
  bool operator<(const ColorMap& o) const {return _d->operator<(*o._d);}

  void resize(int n=10) {_d->resize(n);}
  virtual int count() const {return _d->count();}

  virtual ColorPalette palette() const;
  virtual void setPalette(const ColorPalette& pal);
  virtual void generateColorScale(int n, ColorPalette::Model m,
                                  bool reversed=false, int transparency=255);
  virtual void generateGrayScale(int n, ColorPalette::Model m,
                                 bool reversed=false, int transparency=255);
  virtual void rgbInterpole(int imin, int imax);
  virtual void hsvInterpole(int imin, int imax);
  void setWhiteTransparent(bool b);

  inline void setValues(double min, double max, SamplingParameters::ScaleType t);
  inline void setValues(int imin, int imax, double min, double max, SamplingParameters::ScaleType t);

  void setUpperValue(int i, double v) {_d->setUpperValue(i, v);}
  double upperValue(int i) const {return _d->upperValue(i);}
  double lowerValue(int i) const {return _d->lowerValue(i);}

  const Color& color(int i) const {return _d->color(i);}
  Color& color(int i) {return _d->color(i);}
  const Color& color(double val) const {return _d->color(val);}
  Color& color(double val) {return _d->color(val);}
  void setColor(int i, const Color& c) {return _d->setColor(i, c);}

  int index(double val) const  {return _d->index(val);}
protected:
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
private:
  QSharedDataPointer<ColorMapData> _d;
};

QGPCOREMATH_EXPORT QTextStream& operator>>(QTextStream& s, ColorMap& p);
QGPCOREMATH_EXPORT QTextStream& operator<<(QTextStream& s, const ColorMap& p);

inline void ColorMap::setValues(double min, double max, SamplingParameters::ScaleType t)
{
  _d->setValues(min, max, t);
}

inline void ColorMap::setValues(int imin, int imax, double min, double max, SamplingParameters::ScaleType t)
{
  _d->setValues(imin, imax, min, max, t);
}

} // namespace QGpCoreMath

// Allow passing ColorMap as a QVariant
Q_DECLARE_METATYPE(QGpCoreMath::ColorMap)

#endif // COLORMAP_H
