/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-08-26
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "GeopsySignalHeader.h"

namespace GeopsyCore {

/*!
  \class GeopsySignalHeader GeopsySignalHeader.h
  \brief Brief description of class still missing

  Full description of class still missing

  Version 3 add unitPerCount in the offset table.
    Fast read/write without transformation to count. Current conversion factor is saved.
    Upon next load the the current conversion factor is compared to the saved one, if they
    differ the loaded signal is multiplied/divided.
*/


/*!
  Returns true if header is valid
*/
bool GeopsySignalHeader::isValid()
{
  TRACE;
  if(strcmp( field.tag, GEOPSYSIGNAL_TAG)!=0) {
    // Trying compatibility with version 1
    if(strcmp( compat1.tag, "DBSignals" )!=0) {
      App::log(tr("Cannot read Geopsy Signal file: wrong tag.\n") );
      return false;
    } else { // Compatibility with version 1
      App::log(tr("Compatibility with Geopsy Signal version 1...\n") );
      if(compat1.version==1) {
        if(compat1.offset < 10 + 3 * 4 + compat1.nSignals * (4 + 8) ) {
          App::log(tr("Cannot read Geopsy Signal file: bad offset to first signal.\n") );
          return false;
        }
        // Translate header compat1 to field
        field.nSignals=compat1.nSignals;
        field.offset=compat1.offset;
        field.traceHeaderSize=0;
        field.version=1;
      } else {
        App::log(tr("Cannot read Geopsy Signal file: bad version number.\n") );
        return false;
      }
    }
  } else {
    // Check magic numbers
    if(field.magicFloat!=GEOPSYSIGNAL_MAGICFLOAT  ||
         field.magicInt!=GEOPSYSIGNAL_MAGICINT) {
      // This is the place to implement compatibility checks (e.g. Big/Little endian) if requested.
      App::log(tr("Cannot read Geopsy Signal file: bad magic float. This happens because\n"
                  "the file has been generated with another machine with different number representations.\n"
                  "Please send a request to bug@geopsy.org to improve compatibilty.\n"));
      return false;
    }
    if(field.version==GEOPSYSIGNAL_VERSION) {
      if(field.offset<GEOPSYSIGNAL_HEADERSIZE+field.nSignals*4) {
        App::log(tr("Cannot read Geopsy Signal file: bad offset to first signal.\n") );
        return false;
      }
    } else if(field.version==2){
      if(field.offset<16+8+4*4+field.nSignals*4) {
        App::log(tr("Cannot read Geopsy Signal file: bad offset to first signal.\n") );
        return false;
      }
      field.traceHeaderSize=0;
    } else {
      App::log(tr("Cannot read Geopsy Signal file with version>%1. File version=%2\n")
                  .arg(GEOPSYSIGNAL_VERSION).arg(field.version));
      return false;
    }
  }
  if(field.nSignals<0) {
    App::log(tr("Cannot read Geopsy Signal file: no signals specified.\n") );
    return false;
  }
  return true;
}

/*!
  Returns true if header is valid
*/
bool GeopsySignalTraceHeader::isValid()
{
  TRACE;
  if(field.magicFloat==GEOPSYSIGNAL_MAGICFLOAT) {
    if(field.version==GEOPSYSIGNAL_VERSION) {
      return true;
    } else {
      App::log(1, tr("GeopsySignal, trace header: expecting version 3, found %1.\n").arg(field.version));
      return false;
    }
  } else {
    App::log(1, tr("GeopsySignal, trace header: no magic float found, assuming version 2.\n"));
    field.version=2;
    return true;
  }
}

} // namespace GeopsyCore
