/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-06
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef VORONOINAVIGATOR_H
#define VORONOINAVIGATOR_H

#include <QGpCoreTools.h>
#include "DinverCoreDLLExport.h"
#include "ScaledModels.h"
#include "PdfCurve.h"
#include "Model.h"

namespace DinverCore {

  class DINVERCORE_EXPORT VoronoiNavigator
  {
    TRANSLATIONS("VoronoiNavigator")
  public:
    VoronoiNavigator(const ScaledModels * m);
    ~VoronoiNavigator();

    void cellLimits(int& xMin, int &xMax, ActiveIndex& iMin, ActiveIndex& iMax);
    PdfCurve intersections(int xMin, int xMax);
    PdfCurve intersectionsDebug(int xMin, int xMax);

    inline void setCurrentPoint(const ActiveIndex& index);
    //inline void setCurrentPoint(const int * coordinates);

    const ActiveIndex& currentCell() const {return _currentCell;}
    QVector<ActiveIndex> cellAt(const int * p) const;
    QVector<ActiveIndex> cellAt(const double * p) const;
    int cellCount() const {return _cells->modelCount();}

    void initDistances(int axis);
    void incrementAxis();
    void setCurrentAxis(int axis);
    int currentAxis() const {return _currentAxis;}

    void setValue(int v) {_currentPoint[_currentAxis]=v*_cells->scale(_currentAxis);}
    void setCurrentCell(const ActiveIndex& index) {_currentCell=index;}

    void checkAxisDistances() const;
    void printCurrentPoint(const RealSpace& parameterSpace) const;
    void printPoint(int index, const RealSpace& parameterSpace) const;
    bool checkCurrentPoint(const RealSpace& parameterSpace) const;
  private:
    bool exportGeometry(QString fileName) const;
    void updateDistances(int axis);
    void setCell(int * cells, int nCells);
    void lowerIntersections(PdfCurve& f, double xMinLimit, int * cells, int nCells, double invS2);
    void higherIntersections(PdfCurve& f, double xMaxLimit, int * cells, int nCells, double invS2);

    const ScaledModels * _cells;

    int _currentAxis;
    ActiveIndex _currentCell;
    double * _currentPoint;

    double * _axisDistances;
  };

  inline void VoronoiNavigator::setCurrentPoint(const ActiveIndex& index)
  {
    TRACE;
    _currentCell=index;
    int nd=_cells->parameterCount();
    for(int id=0; id<nd; id++) {
      _currentPoint[id]=_cells->v(id)[_currentCell.value()];
    }
  }

  /*inline void VoronoiNavigator::setCurrentPoint(const int * coordinates)
  {
    TRACE;
    _currentCell=-1;
    int nd=_cells->parameterCount();
    for(int id=0; id < nd; id++ ) {
      _currentPoint[ id ]=_cells->scale(id) * coordinates[id];
    }
  }*/

} // namespace DinverCore

#endif // VORONOINAVIGATOR_H
