/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-06
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>
#include <QGpCoreTools.h>

#include "VoronoiNavigator.h"
#include "RealSpace.h"

namespace DinverCore {

  /*!
    \class VoronoiNavigator VoronoiNavigator.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  VoronoiNavigator::VoronoiNavigator(const ScaledModels * m)
  {
    TRACE;
    _cells=m;

    _currentAxis=-1;
    _currentCell=ActiveIndex::null;

    _axisDistances=new double [_cells->modelCount()];
    _currentPoint=new double [_cells->parameterCount()];
  }

  /*!
    Description of destructor still missing
  */
  VoronoiNavigator::~VoronoiNavigator()
  {
    TRACE;
    delete [] _axisDistances;
    delete [] _currentPoint;
  }

  void VoronoiNavigator::printCurrentPoint(const RealSpace& parameterSpace) const
  {
    int nd=_cells->parameterCount();
    printf("Current point (id scaled grid real)\n");
    for(int id=0; id<nd; id++) {
      int value=qRound(_currentPoint[id]/_cells->scale(id));
      printf("%i %lf %i %lf\n", id, _currentPoint[id], value,
             parameterSpace.variableParameter(id)->realValue(value));
    }
  }

  void VoronoiNavigator::printPoint(int index, const RealSpace& parameterSpace) const
  {
    int nd=_cells->parameterCount();
    printf("Point %i (id scaled grid real)\n", index);
    for(int id=0; id<nd; id++) {
      int value=qRound(_cells->v(id)[index]/_cells->scale(id));
      printf("%i %lf %i %lf\n", id, _cells->v(id)[index], value,
             parameterSpace.variableParameter(id)->realValue(value));
    }
  }

  /*!
    \fn void VoronoiNavigator::setCurrentPoint(const ActiveIndex& index);
    Set current point as one of the cell centers and set current cell to \a index.
  */

  /*!
    Set current axis and calculate the square distances to axis to init the iterative
    computation implemented in incrementAxis(). The current pos must be initialized by
    setCurrentPos().

    ||Vk-Xj||=||Vj-Xj|| defines the intersection of the cell boundary (between cells k and j)
    with the considered axis. k is the current cell. V and X are vectors. Vk and Vj are the
    centers of cells k and j, respectively. Xj is point where the boundary intersects the axis.

    Along the axis i, looking at the cell boundary between cells k and j:

    dk^2 + (( Vk[i] - Xj[i] ) * s[i] )^2=dj^2 + (( Vj[i] - Xj[i] ) * s[i] )^2

    dk^2 + (Vk[i] * s[i] )^2 + (Xj[i] * s[i] )^2 - 2 * Vk[i] * Xj[i] * s[i]^2
                  =dj^2 + (Vj[i] * s[i] )^2 + (Xj[i] * s[i] )^2 - 2 * Vj[i] * Xj[i] * s[i]^2

    dk^2 + (Vk[i] * s[i] )^2=dj^2 + (Vj[i] * s[i] )^2 - 2 * s[i]^2 * Xj[i] (Vj[i] - Vk[i] )

    where dk and dj are the distances from axis to centers of cells.  Vk[i] is the ith coordinate
    of point Vk. s[i] is the scale along axis i.

    2 * Xj[i] * s[i]^2 * (Vk[i] - Vj[i] )=(Vk[i]^2 - Vj[i]^2) * s[i]^2 + (dk^2 - dj^2)

    Pre-calculates all dk and dj. When the point moves along the axis, all distances
    remain the same. All coordinates of current point are used except current axis value.
  */
  void VoronoiNavigator::initDistances(int axis)
  {
    if(axis==_cells->parameterCount()) {
      axis=0;
    }
    _currentAxis=axis;
    double d, dtot;
    double * dist=_axisDistances;
    int nm=_cells->modelCount();
    int nd=_cells->parameterCount();
    int id;
    for(int im=0; im<nm; im++) {
      dtot=0.0;
      for(id=0; id<_currentAxis; id++) {
        d=_currentPoint[id]-_cells->v(id)[im];
        dtot+=d*d;
      }
      for(id++; id<nd; id++) {
        d=_currentPoint[id]-_cells->v(id)[im];
        dtot+=d*d;
      }
      *(dist++)=dtot;
    }
  }

  /*!
    Update the distances to the new \a axis.
  */
  void VoronoiNavigator::updateDistances(int axis)
  {
    double d;
    double xold=_currentPoint[_currentAxis];
    double xnew=_currentPoint[axis];
    double * dist=_axisDistances;
    int nm=_cells->modelCount();
    for(int im=0; im<nm; im++) {
      d=xnew-_cells->v(axis)[im];
      double& distVal=*dist;
      distVal-=d*d;
      d=xold-_cells->v(_currentAxis)[im];
      distVal+=d*d;
      //fprintf(stdout,"Axis Dist=%8lg\n",sqrt(distVal));
      dist++;
    }
    _currentAxis=axis;
  }

  /*!
    Increment axis and update the distances to the new axis
  */
  void VoronoiNavigator::incrementAxis()
  {
    ASSERT(_currentAxis>-1);
    int axis=_currentAxis+1;
    if(axis==_cells->parameterCount()) {
      axis=0;
    }
    updateDistances(axis);
  }

  /*!
    Set current axis and update the distances to the new \a axis.
    \a axis must be from 0 to nd-2
  */
  void VoronoiNavigator::setCurrentAxis(int axis)
  {
    ASSERT(axis<_cells->parameterCount()-1);
    ASSERT(_currentAxis>-1);
    if(axis>=_currentAxis) {  // axis any value different from _currentAxis
      axis++;
    }
    updateDistances(axis);
  }

  /*!
    If Xj[i] is right on the cell border (between Vj and Vk),

      2 * Xj[i] * s[i]^2 * (Vk[i] - Vj[i] )=(Vk[i]^2 - Vj[i]^2) * s[i]^2 + (dk^2 - dj^2)

    Or,

      2 * Xj[i] * s[i]=xLim/dx, dx includes the scaling

      if xLim/(2*dx)<Pk[i]*s[i] && xLim/dx>xMin, move xMin to xLim/dx
      if xLim/(2*dx)>Pk[i]*s[i] && xLim/dx<xMax, move xMax to xLim/dx

    Where Pk is the current point inside cell of Vk, hence,

      xLim < 2 * s[i] * Pk[i] * dx

      if dx>0, xLim/dx < 2 * s[i] * Pk[i]
      if dx<0, xLim/dx > 2 * s[i] * Pk[i]

    We would like to postpone the division by dx only when the limits is really moved. hence,
    we multiply xMin and xMax by dx (division is approximately 5 times slower than multiplication).
    xMin and xMax are multiplied by 2 upon startup.

    if dx>0, the conditions become:

      if xLim/dx<2*Pk[i]*s[i] && xLim>xMin*dx, move xMin to xLim/dx (a)
      if xLim/dx>2*Pk[i]*s[i] && xLim<xMax*dx, move xMax to xLim/dx (b)
      (first condition of (b) is always false, of (a) is always true)

    if dx<0, the conditions become:

      if xLim/dx<2*Pk[i]*s[i] && xLim<xMin*dx, move xMin to xLim/dx (a)
      if xLim/dx>2*Pk[i]*s[i] && xLim>xMax*dx, move xMax to xLim/dx (b)
      (first condition of (a) is always false, of (b) is always true)

    if dx==0, no intersection, hence skip

    if dx>0, the conditions become:

      if xLim>xMin*dx, move xMin to xLim/dx (a)

    if dx<0, the conditions become:

      if xLim>xMax*dx, move xMax to xLim/dx (b)

    _currentCell is k, _axisDistances[ _currentCell ] is dk
    _axisDistances[ j ] is dj

    Thus, we just test the sign of dx and we can directly compare to the current min and max of the cell.

    \a iMin and \a iMax are set to the cell indexes of the two neighbors crossed by the current axis (-1 if
    there are no neighbors).
  */
  void VoronoiNavigator::cellLimits(int& xMin, int& xMax, ActiveIndex& iMin, ActiveIndex& iMax)
  {
    ASSERT(_currentAxis>-1);
    ASSERT(_currentCell.isValid());
    const double * vp=_cells->v(_currentAxis);
    const double * v2p=_cells->v2(_currentAxis);
    double vk=vp[_currentCell.value()];
    double vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
    double xLim, dx;
    double s=_cells->scale(_currentAxis);
    double s2=2.0*s;
    double dxMin=(double)xMin*s2;
    double dxMax=(double)xMax*s2;
    int n=_cells->modelCount();
    iMin=ActiveIndex::null;
    iMax=ActiveIndex::null;

    for(int j=0; j<n; j++) {
      dx=vk-vp[j];
      xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
      if(dx>0.0) {
        if(xLim>dxMin*dx) {
          dxMin=xLim/dx;
          iMin.setValue(j);
        }
      } else if(dx<0.0) {
        if(xLim>dxMax*dx) {
          dxMax=xLim/dx;
          iMax.setValue(j);
        }
      }
    }
    /*
       Distances to axes are updated incrementaly, which is likely to introduce
       some errors in the determination of dxMin and dxMax. By experience, the
       maximum relative error is of the order of 1e-16. This is negligible except
       when rounding limits to integers. By default, we keep as conservative as
       possible, hence limits are rounded with ceil and floor for xMin and xMax,
       respectively. If the current point, which is a valid model (fit with all
       conditions) and which has proved to belong to the current cell, is very
       close to the edge of the cell, the conservative rounding might consider
       the current point as outside. Hence ceil() and floor() are replaced by round()
    */
    double invS=1.0/s;
    double invS2=0.5*invS;
    int pk=qRound(_currentPoint[_currentAxis]*invS);
    if(iMin.isValid()) {
      dxMin*=invS2;
      xMin=qCeil(dxMin);
      if(pk==xMin-1) {
        xMin=pk;
        //App::log(tr("Adjust xMin limit to Pk(%1), dxMin=%2")
        //         .arg(pk).arg(dxMin, 0, 'g', 20) << endl;
      }
    }
    if(iMax.isValid()) {
      dxMax*=invS2;
      xMax=qFloor(dxMax);
      if(pk==xMax+1) {
        xMax=pk;
        //App::log(tr("Adjust xMax limit to Pk(%1), dxMax=%2")
        //         .arg(pk).arg(dxMax, 0, 'g', 20) << endl;
      }
    }
  }

  /*!
    Debug purpose, get the cells containing model.
    Usually there is only one index in the returned vector, except
    if \a p is right on the limit between 2 or more cells.
  */
  QVector<ActiveIndex> VoronoiNavigator::cellAt(const int * p) const
  {
    TRACE;
    QVector<ActiveIndex> belongs;
    double minDistance=std::numeric_limits<double>::infinity();
    int nd=_cells->parameterCount();
    ActiveIndex end;
    end.setValue(_cells->modelCount());
    for(ActiveIndex im; im<end; im++) {
      double distance=0.0, tmp;
      for(int id=0; id<nd; id++) {
        tmp=_cells->v(id)[im.value()]-p[id]*_cells->scale(id);
        distance+=tmp*tmp;
      }
      if(fabs(distance-minDistance)>1e-10*distance) {
        if(distance<minDistance) {
          belongs.clear();
          belongs.append(im);
          minDistance=distance;
        }
      } else {
        belongs.append(im);
      }
    }
    return belongs;
  }

  /*!
    Debug purpose, get the cell(s) containing scaled model.
    Usually there is only one index in the returned vector, except
    if \a p is right on the limit between 2 or more cells.
  */
  QVector<ActiveIndex> VoronoiNavigator::cellAt(const double * p) const
  {
    TRACE;
    QVector<ActiveIndex> belongs;
    double minDistance=std::numeric_limits<double>::infinity();
    int nd=_cells->parameterCount();
    ActiveIndex end;
    end.setValue(_cells->modelCount());
    for(ActiveIndex im; im<end; im++) {
      double distance=0.0, tmp;
      for(int id=0; id<nd; id++) {
        tmp=_cells->v(id)[im.value()]-p[id];
        distance+=tmp*tmp;
      }
      if(fabs(distance-minDistance)>1e-10*distance) {
        if(distance<minDistance) {
          belongs.clear();
          belongs.append(im);
          minDistance=distance;
        }
      } else {
        belongs.append(im);
      }
    }
    return belongs;
  }

  /*!
    Brute comparison of current distance, check for precison leakage
    This function is strictly for debug purposes
  */
  void VoronoiNavigator::checkAxisDistances() const
  {
    TRACE;
    double d, dtot;
    double * dist=_axisDistances;
    int id;
    int nd=_cells->parameterCount();
    ActiveIndex end;
    end.setValue(_cells->modelCount());
    for(ActiveIndex im; im<end; im++) {
      dtot=0.0;
      for(id=0; id<_currentAxis; id++) {
        d=_currentPoint[id]-_cells->v(id)[im.value()];
        dtot+=d*d;
      }
      for(id++; id<nd; id++) {
        d=_currentPoint[id]-_cells->v(id)[im.value()];
        dtot+=d*d;
      }
      if(dtot>0.0) {
        double relErr=(*dist-dtot)/dtot;
        if(fabs(relErr)>1e-10) {
          App::log(tr("  Error in iterative square distance for model %1: %2\n"
                      "      iterative square distance=%3\n"
                      "      true square distance=%4\n")
                    .arg(im.value()).arg(relErr).arg(*dist).arg(dtot));
        }
      } else {
        if(fabs(*dist)>1e-15) {
          App::log(tr("  Error in iterative square distance for model %1:\n"
                      "      iterative square distance=%2\n"
                      "      true square distance=%3\n")
                    .arg(im.value()).arg(*dist).arg(dtot));
        }
      }
      dist++;
    }
  }

  /*!
    Check if current point belongs to current cell
  */
  bool VoronoiNavigator::checkCurrentPoint(const RealSpace& parameterSpace) const
  {
    QVector<ActiveIndex> iGenModel=cellAt(_currentPoint);
    if(!iGenModel.contains(_currentCell)) {
      printCurrentPoint(parameterSpace);
      printPoint(_currentCell.value(), parameterSpace);
      printf("Model effectively NOT generated in cell %i but:\n", _currentCell.value());
      for(QVector<ActiveIndex>::iterator it=iGenModel.begin(); it!=iGenModel.end(); it++) {
        int im=it->value();
        printf("%i:\n", im);
        printPoint(im, parameterSpace);
      }
      printf("\n");
      return false;
    } else {
      return true;
    }
  }

  /*!
    The intersections are returned along current axis set by setAxis().

    The case where xLim==dxMin*dx or xLim=dxMax*dx can be safely ignored.

      \li If the coordinate along axis is the same for two cells, the limit between those two cells
      is parallel to the current axis. If _axisDistances are different the limit is not on the axis.
      On the contrary if the two _axisDistances are equal, the current axis is right on the limit.
      vp and vp2+_axisDistances are the same for the two cells, hence choosing on or the other has
      no influence over the next geometrical computations. If the misfit value are not the same, it
      may have an influence, but anyway the axis is right on the limit and it can on either sides.

      \li If the coordinate along axis is NOT the same for two cells, it matters to choose the correct
      cell. One cell is normally crossed by the axis but others may touch the axis with only one vertex.
      It will results a null step in the pdf curve. It is properly sorted out by PdfCurve::sort() at the end.
  */
  PdfCurve VoronoiNavigator::intersections(int xMin, int xMax)
  {
//#define DEBUG_PDF_BUILD
    PdfCurve f;
    ActiveIndex iMin, iMax;
#ifdef DEBUG_PDF_BUILD
    App::log(tr("intersections(xMin %1 xMax %2)\n").arg(xMin).arg(xMax));
#endif
    ASSERT(_currentAxis>-1);
    ASSERT(_currentCell.isValid());
    const double * vp=_cells->v(_currentAxis);
    const double * v2p=_cells->v2(_currentAxis);
    double vk=vp[_currentCell.value()];
    double vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
    double xLim, dx;
    double s=_cells->scale(_currentAxis);
    double s2=2.0*s;
    double dxMin=(double)xMin*s2;
    double dxMax=(double)xMax*s2;
    int n=_cells->modelCount();
    iMin=ActiveIndex::null;
    iMax=ActiveIndex::null;

    double dxMinLimit=dxMin;
    double dxMaxLimit=dxMax;
    int * lowerCells=new int[n];
    int * higherCells=new int[n];
    int nLowerCells=0, nHigherCells=0;

    for(int j=0; j<n; j++) {
      dx=vk-vp[j];
      xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
      if(dx>0.0) {
        if(xLim>dxMin*dx) {
          dxMin=xLim/dx;
          iMin.setValue(j);
        }
        lowerCells[nLowerCells++]=j;
      } else if(dx<0.0) {
        if(xLim>dxMax*dx) {
          dxMax=xLim/dx;
          iMax.setValue(j);
        }
        higherCells[nHigherCells++]=j;
      }
    }
    double invS2=1.0/s2;
#ifdef DEBUG_PDF_BUILD
    App::log(tr("Building PDF...\n"));
    App::log(tr("  mid append %1 %2\n").arg(invS2*dxMin).arg(_currentCell.value()));
#endif
    f.append(invS2*dxMin, _currentCell);
#ifdef DEBUG_PDF_BUILD
    App::log(tr("  mid append %1 %2\n").arg(invS2*dxMax).arg(iMax.value()));
#endif
    f.append(invS2*dxMax, iMax);
    ActiveIndex originalCell=_currentCell;
    if(iMin.isValid()) {
      _currentCell=iMin;
      lowerIntersections(f, dxMinLimit, lowerCells, nLowerCells, invS2);
    }
    if(iMax.isValid()) {
      _currentCell=iMax;
      higherIntersections(f, dxMaxLimit, higherCells, nHigherCells, invS2);
    }
    _currentCell=originalCell;
    delete [] lowerCells;
    delete [] higherCells;
    f.sort();
    return f;
  }

  /*!
    For intersections
  */
  void VoronoiNavigator::lowerIntersections(PdfCurve& f, double dxMinLimit, int * cells, int nCells, double invS2)
  {
    const double * vp=_cells->v(_currentAxis);
    const double * v2p=_cells->v2(_currentAxis);
    double vk=vp[_currentCell.value()];
    double vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
    double xLim, dx;
    int n=_cells->modelCount();

    double dxMin=dxMinLimit;
    int * subCells=new int[n];
    int nSubCells=0;
    ActiveIndex iMin=ActiveIndex::null;
    int j;
    for(int ji=0; ji<nCells; ji++) {
      j=cells[ji];
      dx=vk-vp[j];
      xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
      if(dx>0.0) {
        if(xLim>dxMin*dx) {
          dxMin=xLim/dx;
          iMin.setValue(j);
        }
        subCells[nSubCells]=j;
        nSubCells++;
      }
    }
#ifdef DEBUG_PDF_BUILD
    App::log(tr("  lower prepend %1 %2\n").arg(invS2*dxMin).arg(_currentCell.value()));
#endif
    f.prepend(invS2*dxMin, _currentCell);
    if(iMin.isValid()) {
      _currentCell=iMin;
      lowerIntersections(f, dxMinLimit, subCells, nSubCells, invS2);
    }
    delete [] subCells;
  }

  /*!
    For intersections
  */
  void VoronoiNavigator::higherIntersections(PdfCurve& f, double dxMaxLimit, int * cells, int nCells, double invS2)
  {
    const double * vp=_cells->v(_currentAxis);
    const double * v2p=_cells->v2(_currentAxis);
    double vk=vp[_currentCell.value()];
    double vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
    double xLim, dx;
    int n=_cells->modelCount();

    double dxMax=dxMaxLimit;
    int * subCells=new int[n];
    int nSubCells=0;
    ActiveIndex iMax=ActiveIndex::null;
    int j;
    for(int ji=0; ji<nCells; ji++) {
      j=cells[ji];
      dx=vk-vp[j];
      xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
      if(dx<0.0) {
        if(xLim>dxMax*dx) {
          dxMax=xLim/dx;
          iMax.setValue(j);
        }
        subCells[nSubCells]=j;
        nSubCells++;
      }
    }
    if(iMax.isValid()) {
#ifdef DEBUG_PDF_BUILD
      App::log(tr("  higher append %1 %2\n").arg(invS2*dxMax).arg(iMax.value()));
#endif
      f.append(invS2*dxMax, iMax);
      _currentCell=iMax;
      higherIntersections(f, dxMaxLimit, subCells, nSubCells, invS2);
    } else {
#ifdef DEBUG_PDF_BUILD
      App::log(tr("  higher append %1 %2\n").arg(invS2*dxMax).arg(-1));
#endif
      f.append(invS2*dxMax, ActiveIndex::null);
    }
    delete [] subCells;
  }

  /*!
    The intersections are returned along current axis set by setAxis()
  */
  PdfCurve VoronoiNavigator::intersectionsDebug(int xMin, int xMax)
  {
    PdfCurve f;
    ActiveIndex iMin, iMax;

    ASSERT(_currentAxis>-1);
    ASSERT(_currentCell.isValid());
    const double * vp=_cells->v(_currentAxis);
    const double * v2p=_cells->v2(_currentAxis);
    double vk=vp[_currentCell.value()];
    double vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
    double xLim, dx;
    double s=_cells->scale(_currentAxis);
    double s2=2.0*s;
    double dxMin=(double)xMin*s2;
    double dxMax=(double)xMax*s2;
    int n=_cells->modelCount();
    iMin=ActiveIndex::null;
    iMax=ActiveIndex::null;
    ActiveIndex ambiguityIndexMin=ActiveIndex::null;
    ActiveIndex ambiguityIndexMax=ActiveIndex::null;

    for(int j=0; j<n; j++) {
      dx=vk-vp[j];
      xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
      if(dx>0.0) {
        if(xLim>dxMin*dx) {
          if(xLim<=0.0) {
            xLim=0.0;
          }
          dxMin=xLim/dx;
          iMin.setValue(j);
        }
      } else if(dx<0.0) {
        if(xLim>dxMax*dx) {
          if(xLim>=0.0) {
            xLim=0.0;
          }
          dxMax=xLim/dx;
          iMax.setValue(j);
        }
      }
    }
    ActiveIndex originalCell=_currentCell;
    double invS2=0.5/s;
    while(iMin.isValid()) {
      f.prepend(invS2*dxMin, _currentCell);
      _currentCell=iMin;
      dxMin=(double)xMin*s2;
      vk=vp[_currentCell.value()];
      vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
      iMin=ActiveIndex::null;
      ambiguityIndexMin=ActiveIndex::null;

      for(int j=0; j<n; j++) {
        dx=vk-vp[j];
        xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
        if(dx>0.0) {
          if(xLim>dxMin*dx) {
            if(xLim<=0.0) {
              xLim=0.0;
            }
            dxMin=xLim/dx;
            iMin.setValue(j);
          }
        }
      }
    }
    // First point at minimum X (open cell)
    // invS2*dxMin==xMin but left like that for comparison with intersections()
    f.prepend(invS2*dxMin, _currentCell);
    while(iMax.isValid()) {
      f.append(invS2*dxMax, iMax);
      _currentCell=iMax;
      dxMax=(double)xMax*s2;
      vk=vp[_currentCell.value()];
      vk2dk2=v2p[_currentCell.value()]+_axisDistances[_currentCell.value()];
      iMax=ActiveIndex::null;
      ambiguityIndexMax=ActiveIndex::null;

      for(int j=0; j<n; j++) {
        dx=vk-vp[j];
        xLim=vk2dk2-(v2p[j]+_axisDistances[j]);
        if(dx<0.0) {
          if(xLim>dxMax*dx) {
            if(xLim>=0.0) {
              xLim=0.0;
            }
            dxMax=xLim/dx;
            iMax.setValue(j);
          }
        }
      }
    }
    // Last point at maximum X (open cell),
    // invS2*dxMax==xMax but left like that for comparison with intersections()
    f.append(invS2*dxMax, ActiveIndex::null);
    _currentCell=originalCell;
    f.sort();
    return f;
  }

  bool VoronoiNavigator::exportGeometry(QString fileName) const
  {
    QFile f(fileName);
    if(!f.open(QIODevice::WriteOnly)) {
      App::log(tr("Cannot open file '%1' for writing.").arg(fileName));
      return false;
    }
    QTextStream s(&f);
    s.setFieldWidth(20);
    s.setRealNumberNotation(QTextStream::SmartNotation);
    s.setRealNumberPrecision(10);
    int nm=_cells->modelCount();
    int nd=_cells->parameterCount();
    ASSERT(nd>0);
    for(int im=0; im<nm; im++) {
      s << _cells->v(0)[im];
      for(int id=1; id<nd; id++) {
        s << " " << _cells->v(id)[im];
      }
      s << im << "\n";
    }
    return true;
  }

} // namespace DinverCore
