/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-09-06
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "StationPair.h"

namespace ArrayCore {

  StationPair::StationPair()
  {
    _s1=0;
    _s2=0;
  }

  StationPair::StationPair(const StationPair& o)
  {
    _s1=o._s1;
    _s2=o._s2;
    _distance=o._distance;
    _azimuth=o._azimuth;
    _name=o._name;
  }

  void StationPair::setStations(const ArraySelection& array, int index1, int index2)
  {
    TRACE;
    _s1=index1;
    _s2=index2;
    const StationSignals * ss1=array.at(_s1);
    const StationSignals * ss2=array.at(_s2);
    _name=ss1->name()+"-"+ss2->name();
    const Point& p1=ss1->coordinates();
    const Point& p2=ss2->coordinates();
    _distance=p1.distanceTo(p2);
    _azimuth.set(p2.x()-p1.x(), p2.y()-p1.y(), _distance);
    _azimuth.initRadians();
    _azimuth.initDegrees();
    if(_azimuth.sin()<0.0) _azimuth.mirror();
  }

  void StationPair::setStations(const NamedPoint& stat1, const NamedPoint& stat2)
  {
    TRACE;
    _s1=-1;
    _s2=-1;
    _name=stat1.name()+"-"+stat2.name();
    _distance=stat1.distanceTo(stat2);
    _azimuth.set(stat2.x()-stat1.x(), stat2.y()-stat1.y(), _distance);
    _azimuth.initRadians();
    _azimuth.initDegrees();
    if(_azimuth.sin()<0.0) _azimuth.mirror();
    ASSERT(_distance>0.0);
  }

  bool StationPair::lessThan(const StationPair& p1, const StationPair& p2)
  {
    return p1.distance()<p2.distance();
  }

  NamedPoint StationPair::coArray() const
  {
    NamedPoint p;
    p.setName(_name);
    p.setX(_distance*_azimuth.cos());
    p.setY(_distance*_azimuth.sin());
    return p;
  }

} // namespace ArrayCore
