/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-11-17
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleigh.h"
#include "HRFKRayleighEllipticity.h"
#include "FKCrossSpectrum.h"
#include "FKGridSearch.h"
#include "FKPower.h"
#include "FKSteeringOneComponent.h"
#include "FKSteeringTwoComponentRayleighRadial.h"

namespace ArrayCore {

  /*!
    \class HRFKRayleigh HRFKRayleigh.h
    \brief Rayleigh high resolution FK power

    Rayleigh high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
  */

  HRFKRayleigh::HRFKRayleigh(FKCache * gridCache)
    : AbstractFKFunction(gridCache)
  {
    delete _steering;
    _steering=new FKSteeringTwoComponentRayleighRadial;
    _steering->setArray(gridCache->array());
    static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering)->resize();

    _RoverN=0.0;
    _ellipticity=0.0;
    _kxkyCacheCount=0;
  }

  void HRFKRayleigh::resetCrossSpectrum()
  {
    _crossSpectrum->resetRotatedMatrices();
  }

  void HRFKRayleigh::addCrossSpectrum()
  {
    _crossSpectrum->addRotatedRayleigh();
  }

  void HRFKRayleigh::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRotatedMatrices(nBlocks);
  }

  bool HRFKRayleigh::invertCrossSpectrum()
  {
    return _crossSpectrum->invertRotatedMatrices();
  }

  void HRFKRayleigh::setGrid(FKGridSearch * g, double step, double size)
  {
    g->setGrid(-size, size, step, -size, size, step, -0.5*M_PI, 0.5*M_PI, M_PI/20.0);
    setKxKySteering(g);
    _kxkyCacheCount=g->nxy();
  }

  double HRFKRayleigh::value(const Point& kell, int index) const
  {
    if(isInsideLimits(kell)) {
      double ell=::tan(kell.z());
      FKSteeringTwoComponentRayleighRadial * s=static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering);
      s->initValue(FKSteeringOneComponent::cache(_gridCache, index%_kxkyCacheCount), ell);
      FKPower p(s);
      p.setCaponValue(_crossSpectrum->rotatedMatrix(kell));
      return p.value()*p.value()*ell*ell;
    } else {
      return -1.0;
    }
  }

  double HRFKRayleigh::value(const Point& kell) const
  {
    if(isInsideLimits(kell)) {
      double ell=::tan(kell.z());
      FKSteeringTwoComponentRayleighRadial * s=static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering);
      s->initValue(kell, ell);
      FKPower p(s);
      p.setCaponValue(_crossSpectrum->rotatedMatrix(kell));
      return p.value()*p.value()*ell*ell;
    } else {
      return -1.0;
    }
  }

  double HRFKRayleigh::ellipticity(const Point& kell, bool& ok)
  {
    const ComplexMatrix& covmat=_crossSpectrum->rotatedMatrix(kell);
    FKGridSearch grid;
    grid.setEllipticityPrecision(0);
    double ellh=0.0, ellz=0.0;

    // TODO: implement a Newton-Raphson, derivative of power versus ellipticity
    //       is possible even for RTBF because k is held constant.
    //       Function is smooth even for real datasets
    HRFKRayleighRadialEllipticity * fh;
    fh=new HRFKRayleighRadialEllipticity(_gridCache);
    grid.setFunction(fh);
    fh->setK(kell);
    fh->setCrossSpectrum(&covmat);
    if(kell.z()<0.0) {
      grid.setGrid(-0.5*M_PI, 0.0, M_PI/40.0);
    } else {
      grid.setGrid(0.0, 0.5*M_PI, M_PI/40.0);
    }
    grid.localMax(INT_MAX, 0.0, 0.0);
    switch(grid.maximaCount()) {
    case 0:
      // Happens rather often at high frequencies affected by aliasing.
      //APP_LOG(2, tr("No maximum found for e_h with the same sign as e_s.\n"));
      ok=false;
      _RoverN=std::numeric_limits<double>::infinity();
      _ellipticity=std::numeric_limits<double>::quiet_NaN();;
      return _ellipticity;
    case 1:
      ellh=::tan(grid.pos().x());
      break;
    default:
      ASSERT(false);
    }

    HRFKRayleighVerticalEllipticity * fz;
    fz=new HRFKRayleighVerticalEllipticity(_gridCache);
    grid.setFunction(fz);
    fz->setK(kell);
    fz->setCrossSpectrum(&covmat);
    if(kell.z()<0.0) {
      grid.setGrid(-0.5*M_PI, 0.0, M_PI/40.0);
    } else {
      grid.setGrid(0.0, 0.5*M_PI, M_PI/40.0);
    }
    grid.localMax(INT_MAX, 0.0, 0.0);
    switch(grid.maximaCount()) {
    case 0:
      // Happens rather often at high frequencies affected by aliasing.
      //APP_LOG(2, tr("No maximum found for e_z with the same sign as e_s.\n"));
      ok=false;
      _RoverN=std::numeric_limits<double>::infinity();
      _ellipticity=std::numeric_limits<double>::quiet_NaN();;
      return _ellipticity;
    case 1:
      ellz=::tan(grid.pos().x());
      break;
    default:
      ASSERT(false);
    }
    double ellRatio=ellz/ellh;
    if(ellRatio>=1.0) { // They must have the same sign and fabs(ellz)>fabs(ellh)
      _RoverN=::sqrt(ellRatio)-1.0;
      _ellipticity=ellh*(_RoverN+1.0);  // Which is equivalent to sqrt(ellz*ellh)
                                        // It avoids double computation of sqrt.
      // Check that the two ways of getting ellipticity provides the same results
      if(fabs(kell.z()-::atan(_ellipticity))>Angle::degreesToRadians(1.0)) {
        // Happens only for a few cases per run
        APP_LOG(1, tr("More than 1 deg. between e_s and sqrt(e_z*e_h): %1 deg.\n")
                .arg(Angle::radiansToDegrees(fabs(kell.z()-::atan(_ellipticity)))));
        ok=false;
        _ellipticity=std::numeric_limits<double>::quiet_NaN();
      } else {
        ok=true;
      }
    } else {
      ASSERT(false);
      ok=false;
      _RoverN=std::numeric_limits<double>::infinity();
      _ellipticity=std::numeric_limits<double>::quiet_NaN();
    }
    return _ellipticity;
  }

  double HRFKRayleigh::noise(const Point&) const
  {
    return _crossSpectrum->array().count()*_RoverN;
  }

  double HRFKRayleigh::power(double optimumPower) const
  {
    if(_ellipticity!=0.0) {
      return ::pow(optimumPower/(_ellipticity*_ellipticity), 0.5);
    } else {
      return 0.0;
    }
  }

  /*!
    Ouput curves to /tmp/ellpowcurves

    Before runnning:
      rm /tmp/ellpowcurves
    After runing:
      mkdir /tmp/ell
      cd /tmp/ell
      split ../ellpowcurves -l 42
      for f in $(ls); do ( echo "1 11";awk '/#/{print $2,$3,$4}' $f ) | figue -t -nobugreport -export ../text.page; cat $f | figue -c -nobugreport -export ../curve.page; figue ../text.page ../curve.page -nobugreport; done
  */
  void HRFKRayleigh::printEllipticityCurve(const FKGridSearch& grid, double ell) const
  {
    static QMutex mutex;
    mutex.lock();
    QFile f("/tmp/ellpowcurves");
    f.open(QIODevice::Append);
    QTextStream s(&f);
    s << "# " << Angle::radiansToDegrees(ell) << Qt::endl;
    for(int i=0; i<grid.nxyz(); i++) {
      s << Angle::radiansToDegrees(grid.coordinates(i).x()) << " " << grid.value(i) << Qt::endl;
    }
    f.close();
    mutex.unlock();
  }

} // namespace ArrayCore

