/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-05-15
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectLove.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKSteeringTwoComponentTransverse.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectLove HRFKDirectLove.h
    \brief Transverse high resolution FK power with direct steering of a 2-component cross spectrum

    Transverse high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is made of raw component E and N.
    Projection on the transverse direction is included in the steering matrix.
  */

  HRFKDirectLove::HRFKDirectLove(FKCache * cache)
    : HRFKTransverse(cache)
  {
    delete _steering;
    _steering=new FKSteeringTwoComponentTransverse;
    _steering->setArray(cache->array());
    static_cast<FKSteeringTwoComponentTransverse *>(_steering)->resize();
  }

  void HRFKDirectLove::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
  }

  void HRFKDirectLove::addCrossSpectrum()
  {
    _crossSpectrum->addHorizontal();
  }

  void HRFKDirectLove::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
  }

  bool HRFKDirectLove::invertCrossSpectrum()
  {
    return _crossSpectrum->invert();
  }

  bool HRFKDirectLove::initGrid(int n)
  {
    if(_gridCache->isEmpty(FKCache::TwoComponentTransverse)) {
      App::log(tr("Caching steering vectors for 2-component transverse ... (%1 values)\n").arg(n) );
      _gridCache->resize(FKCache::TwoComponentTransverse, n);
      _gridCache->reportCacheVolume(FKCache::TwoComponentTransverse, n,
                                    FKSteeringTwoComponentTransverse::sizeFactor);
      return true;
    } else {
      return false;
    }
  }

  void HRFKDirectLove::initGrid(const Point& k, int index)
  {
    FKSteeringTwoComponentTransverse::init(_gridCache, index , k);
  }

  double HRFKDirectLove::value(const Point& k, int index) const
  {
    if(isInsideLimits(k)) {
      FKPower p(&FKSteeringTwoComponentTransverse::cache(_gridCache, index));
      p.setCaponValue(_crossSpectrum->matrix());
      return p.value();
    } else {
      return -1.0;
    }
  }

  double HRFKDirectLove::value(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKSteeringTwoComponentTransverse * s=static_cast<FKSteeringTwoComponentTransverse *>(_steering);
      s->initValue(k);
      FKPower p(s);
      p.setCaponValue(_crossSpectrum->matrix());
      return p.value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

