/***************************************************************************
**
**  This file is part of gptarget.
**
**  gptarget is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gptarget is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-28
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "gptargetVersion.h"
#include "gptargetInstallPath.h"
#include "TargetAdd.h"
#include "TargetExtract.h"
#include "TargetMerge.h"

PACKAGE_INFO("gptarget", GPTARGET)

ApplicationHelp * help();

int main(int argc, char ** argv)
{
  enum Mode {Undefined, Add, Delete, Extract, Merge, Split};
  Mode mode=Undefined;

  if(argc>1) {
    switch (argv[1][0]) {
    case 'A':
      mode=Add;
      break;
    case 'D':
      mode=Delete;
      break;
    case 'E':
      mode=Extract;
      break;
    case 'M':
      mode=Merge;
      break;
    case 'S':
      mode=Split;
      break;
    default:
      break;
    }
  }
  if(mode!=Undefined) {
    for(int i=2; i<argc; i++) {
      argv[i-1]=argv[i];
    }
    argc--;
  }
  CoreApplication a(argc, argv, help);

  AbstractTarget * t=nullptr;

  switch(mode) {
  case Add:
    t=new TargetAdd();
    break;
  case Delete:
    // Temporary message
    App::log(tr("gptarget: mode 'Delete' currently not supported. Request it to the developers if needed.\n") );
    return 2;
  case Extract:
    t=new TargetExtract();
    break;
  case Merge:
    t=new TargetMerge();
    break;
  case Split:
    // Temporary message
    App::log(tr("gptarget: mode 'Split' currently not supported. Request it to the developers if needed.\n") );
    return 2;
  case Undefined:
    App::log(tr("gptarget: mode not specified, see -help\n") );
    return 2;
  }
  t->setOptions(argc, argv);
  int exitCode=t->execute();
  delete t;
  return exitCode;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "MODE [OPTIONS] [FILE.target]" );
  h->setComments( "Manipulation of dinver targets through command line. This tool "
                  "can add, delete, or extract curves to or from FILE.target file. "
                  "If FILE.target does not exist, it is created. This tool currently "
                  "support only dispersion curves but can be extended to other "
                  "target types upon request. For 'Merge' mode many FILE.target are "
                  "accepted" );
  h->addGroup("Mode", "mode");
  h->addOption("A","Add information to FILE.target (see '-h add')");
  h->addOption("D","Delete information from FILE.target (see '-h delete')");
  h->addOption("E","Extract information from FILE.target (see '-h extract')");
  h->addOption("M","Merge various 1D targets into a 2D target");
  h->addOption("S","Split a 2D target in various 1D targets");
  h->addGroup("Add", "add");
  h->addOption("-dispersion-rayleigh <INDEX>","Adds dispersion curves read from stdin as Rayleigh with "
                                              "mode INDEX (0 is fundamental). Format: frequency(Hz) "
                                              "slowness(s/m) log stddev [weight(0 for invalid)]. "
                                              "Blanks or lines starting with '#' delimit various curves.");
  h->addOption("-dispersion-love <INDEX>","Adds dispersion curves read from stdin as Love with mode "
                                          "mode INDEX (0 is fundamental). Format: frequency(Hz) "
                                          "slowness(s/m) log stddev [weight(0 for invalid)]. "
                                          "Blanks or lines starting with '#' delimit various curves.");
  h->addOption("-group","Set Group slowness (default is Phase).");
  h->addOption("-mt","Adds magneto-telluric curves read from stdin. "
                     "Format: frequency(Hz) real(V/m/T) imaginary(V/m/T)[ stddev(V/m/T) weight(0 for invalid)]. "
                     "Blanks or lines starting with '#' delimit various curves.");
  h->addOption("-p \"x y z\"", "Set position of target");
  h->addOption("-weight <WEIGHT>","Set misfit weight for dispersion curves (default=1).");
  h->addOption("-min-misfit <MISFIT>","Set minimum misfit for dispersion curves (default=0).");
  h->addGroup("Delete", "delete");
  h->addOption("-disp <INDEX>","Deletes dispersion curve with INDEX.");
  h->addGroup("Extract", "extract");
  h->addOption("-disp <INDEX>","Extracts dispersion curve with INDEX.");
  h->addOption("-autocorr <INDEX>","Extracts autocorrelation curve with INDEX.");
  h->addOption("-ring <INDEX>","Extracts autocorrelation curves from ring INDEX (must be used with -autocorr, default=0).");
  h->addGroup("Merge", "merge");
  h->addOption("-o <TARGET>","Export 2D target to TARGET file.");
  h->addExample("cat my.disp | gptarget A -dispersion-rayleigh 0 my.target","Add the dispersion curve contains in a column text file to 'my.target' assigning it to the phase Rayleigh fundamental mode.");
  h->addExample("gptarget E -disp 0 my.target","Extracts the dispersion curve of fundamental mode contained in target file 'my.target'.");
  return h;
}
