/***************************************************************************
**
**  This file is part of gpell.
**
**  gpell is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpell is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-03
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include <QGpCoreWave.h>

#include "EllipticityReader.h"
#include "gpellVersion.h"
#include "gpellInstallPath.h"

PACKAGE_INFO("gpell", GPELL)

ApplicationHelp * help();

int main(int argc, char ** argv)
{
  CoreApplication a(argc, argv, help);

  // Options
  EllipticityReader reader;
  if(reader.setOptions(argc, argv) && reader.read(argc, argv) ) {
    return 0;
  } else {
    return 2;
  }
}

ApplicationHelp * help()
{
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] [FILE]" );
  h->setComments( "Compute ellipticity curve for layered models given through stdin or FILE.\n\n"
                 "Format for layered models:\n"
                 + Seismic1DModel::formatHelp() +
                 "\n\n"
                 "Quality factors are not mandatory. If not specified, pure elastic computation is performed. Any number of models "
                 "can be given as input.");
  h->addGroup("Gpell","gpell");
  h->addOption("-step <s>","Step between frequencies (default=1.025 for log scale)");
  h->addOption("-n <count>","Number of samples (default=100)");
  h->addOption("-R <n modes>","Number of Rayleigh modes (default=1)");
  h->addOption("-one-mode","Instead of outputting all modes (see option '-R'), output only the highest one.");
  h->addOption("-abs","Output abs(ellipticities).");
  h->addOption("-angle","Output atan(ellipticity) in deg.");
  h->addOption("-angle-deg","Output atan(ellipticity) in deg.");
  h->addOption("-angle-rad","Output atan(ellipticity) in rad.");
  h->addOption("-plot", "Insert '#' everytime the ellipticity sign changes with a high absolute amplitude. "
                        "It is more convenient for plotting.");
  h->addOption("-love <alpha>","Adds a Love contribution, 0<=alpha<1 (default=0)");
  h->addOption("-s <sampling>","Defines the sampling type:\n"
                              "  period     regular sampling in period\n"
                              "  frequency  regular sampling in frequency\n"
                              "  log        regular sampling in log(frequency) (default)");
  h->addOption("-min <min>","Minimum of range for ellipticity curve (default=0.2)");
  h->addOption("-max <max>","Maximum of range for ellipticity curve (default=20)");
  h->addOption("-n <count>","Number of samples (default=100)");
  h->addOption("-p","Output frequency peaks only");
  h->addOption("-c","Output curves only");
  h->addOption("-pc","Output curves only with peak refinements");
  h->addOption("-half-space <index>", "Take Vp, Vs, Qp and Qs from layer index and compute the ellipticity for a half-space (from Boore and Toksoz 1969).");
  h->addExample("gpell < test.model", "Calculate fundamental ellipticity curve from 0.2 Hz to 20 Hz for model 'test.model'.");
  h->addExample("gpell < test.model | figue -c", "Calculate the same ellipticity curve and plot it.");
  h->addExample("gpell < test.model | figue -c -m ell.mkup", "Show the same ellipticity curve on a log-log plot. 'ell.mkup' is a tar.gz file containing an xml description of the graphic format, it can be generated from figue's interface.");
  return h;
}
