/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-10-20
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef IMAGELAYER_H
#define IMAGELAYER_H

#include <SciFigs.h>
#include "QGpGuiMathDLLExport.h"

namespace QGpGuiMath {

  class ImageLayerProperties;

  class QGPGUIMATH_EXPORT ImageLayer: public GraphContentLayer
  {
    Q_OBJECT
    Q_PROPERTY(QString imageFile READ fileName WRITE setFileName)
    Q_PROPERTY(double xScale READ xScale WRITE setXScale)
    Q_PROPERTY(double yScale READ yScale WRITE setYScale)
    Q_PROPERTY(double xOrigin READ xOrigin WRITE setXOrigin)
    Q_PROPERTY(double yOrigin READ yOrigin WRITE setYOrigin)
  public:
    ImageLayer(AxisWindow * parent=nullptr);
    ~ImageLayer();

    virtual const QString& xml_tagName() const {return xmlImageLayerTag;}
    static const QString xmlImageLayerTag;

    void loadImage(QString fileName);
    void loadImage();
    void colorHistogram(Rect r);
    enum TrackingModes{Scale=0, Histogram};
    virtual bool trackRectangle(int id, double rx1,double ry1, double rx2, double ry2, Qt::KeyboardModifiers m);
    virtual bool mouseReleaseEvent (QMouseEvent * e, int id=-1);
    virtual void toggleTrackingAction(bool checked, int id=-1);
    virtual Rect boundingRect() const;
    void grayFilter(int tol,int threshold);
    void lowPassFilter(int hThres, int sThres, int vThres);
    void highPassFilter(int hThres, int sThres, int vThres);
    void redPassFilter();

    QString fileName() const {return _fileName;}
    void setFileName(QString s) {_fileName=s;}

    const Point2D& origin() const {return _origin;}
    Point2D& origin() {return _origin;}
    void setOrigin(const Point2D& o) {_origin=o;}

    const Point2D& scale() const {return _scale;}
    Point2D& scale() {return _scale;}
    void setScale(const Point2D& s) {_scale=s;}

    double xScale() const {return _scale.x();}
    void setXScale(double s) {_scale.setX(s);}
    double yScale() const {return _scale.y();}
    void setYScale(double s) {_scale.setY(s);}

    double xOrigin() const {return _origin.x();}
    void setXOrigin(double s) {_origin.setX(s);}
    double yOrigin() const {return _origin.y();}
    void setYOrigin(double s) {_origin.setY(s);}

    void setScaling();

    QRgb * image(int& n);

    virtual bool hasProperties() {return true;}
    virtual void addProperties(PropertyProxy * pp);
    virtual void removeProperties(PropertyProxy * pp);
    virtual void properties(PropertyWidget * w) const;
    virtual void setProperty(uint wid, int pid, QVariant val);

    class QGPGUIMATH_EXPORT ReferencePoint: public XMLClass
    {
    public:
      ReferencePoint() : XMLClass() {}
      ReferencePoint(const ReferencePoint& o)
        : XMLClass(), _name(o._name), _image(o._image), _real(o._real) {}

      virtual const QString& xml_tagName() const {return xmlReferencePointTag;}
      static const QString xmlReferencePointTag;

      void setName(const QString& n) {_name=n;}
      const QString& name() const {return _name;}

      void setImage(const QPoint& p) {_image=p;}
      const QPoint& image() const {return _image;}
      QPoint& image() {return _image;}

      void setReal(const Point2D& p) {_real=p;}
      const Point2D& real() const {return _real;}
      Point2D& real() {return _real;}
    protected:
      virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
      virtual XMLMember xml_member(XML_MEMBER_ARGS);
      virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
    private:
      QString _name;
      QPoint _image;
      Point2D _real;
    };

    void setReferences(const QList<ReferencePoint>& ref) {_references=ref;}
    const QList<ReferencePoint>& references() const {return _references;}

    static void scaling(const QList<ReferencePoint> ref, Point2D& origin, Point2D& scale);
  protected:
    virtual void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const;
    virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
    virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
    virtual XMLMember xml_member(XML_MEMBER_ARGS);
    virtual bool xml_polish(XML_POLISH_ARGS);
  private:
    inline QPoint r2i(const Point2D& p) const;

    QImage _trueImage;
    QTransform _transform;
    QString _fileName;
    Point2D _origin;
    Point2D _scale;
    QList<ReferencePoint> _references;
    static uint _tabImage;
  signals:
    void editedColors(QRgb * colors, int numC);
    void pointPicked(QPoint p);
  };

  inline QPoint ImageLayer::r2i(const Point2D& p) const
  {
    return QPoint((int)round((p.x()-_origin.x())/_scale.x()),
                  (int)round((p.y()-_origin.y())/_scale.y()));
  }

} // namespace QGpGuiMath

#endif // IMAGELAYER_H
