/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-25
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AUTOCORRRING_H
#define AUTOCORRRING_H

#include <QGpCoreTools.h>

#include "FactoryPoint.h"
#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class QGPCOREWAVE_EXPORT AutocorrRing : public XMLClass
{
public:
  inline AutocorrRing();
  inline AutocorrRing(const AutocorrRing& o);
  inline AutocorrRing(double rMin, double rMax);

  inline AutocorrRing& operator=(const AutocorrRing& o);
  inline bool operator==(const AutocorrRing& o) const;
  inline bool operator<(const AutocorrRing& o) const;

  virtual const QString& xml_tagName() const {return xmlAutocorrRingTag;}
  static const QString xmlAutocorrRingTag;

  inline void setRadii(double rMin, double rMax);
  double minRadius() const {return _minRadius;}
  double maxRadius() const {return _maxRadius;}
  void round(double step);

  QString toUserString() const;
  QString toString() const;
  bool fromString(LineParser& p, int startIndex=0);
protected:
  void setMinRadius(double r) {_minRadius=r;}
  void setMaxRadius(double r) {_maxRadius=r;}
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
private:
  double _minRadius, _maxRadius;
};

inline AutocorrRing::AutocorrRing()
  : XMLClass()
{
  _minRadius=1.0;
  _maxRadius=2.0;
}

inline AutocorrRing::AutocorrRing(const AutocorrRing& o)
  : XMLClass()
{
  setRadii(o.minRadius(), o.maxRadius());
}

inline AutocorrRing::AutocorrRing(double rMin, double rMax)
  : XMLClass()
{
  setRadii(rMin, rMax);
}

inline AutocorrRing& AutocorrRing::operator=(const AutocorrRing& o)
{
  TRACE;
  _minRadius=o._minRadius;
  _maxRadius=o._maxRadius;
  return *this;
}

inline bool AutocorrRing::operator==(const AutocorrRing& o) const
{
  TRACE;
  return _minRadius==o._minRadius && _maxRadius==o._maxRadius;
}

inline bool AutocorrRing::operator<(const AutocorrRing& o) const
{
  TRACE;
  return _minRadius<o._minRadius || (_minRadius==o._minRadius && _maxRadius<o._maxRadius);
}

inline void AutocorrRing::setRadii(double rMin, double rMax)
{
  if(rMin < rMax) {
    _minRadius=rMin;
    _maxRadius=rMax;
  } else {
    _maxRadius=rMin;
    _minRadius=rMax;
  }
}

} // namespace QGpCoreWave

#endif // AUTOCORRRING_H
