/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-22
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef VALUE_H
#define VALUE_H

#include <QGpCoreTools.h>

#include "Angle.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  template <class numberType>
  class QGPCOREMATH_EXPORT Value
  {
  public:
    Value() {_valid=false;}
    Value(const numberType& val) {_valid=true; _value=val;}
    Value(const Value<numberType>& o) {_valid=o._valid; _value=o._value;}
    ~Value() {}

    void operator=(const Value& o) {_valid=o._valid; _value=o._value;}
    void operator=(const numberType& val) {_valid=true; _value=val;}

    const numberType& value() const {return _value;}
    void setValue(const numberType& val) {_value=val; _valid=true;}

    void setValid(bool v) {_valid=v;}
    bool isValid() const {return _valid;}

  private:
    numberType _value;
    bool _valid;
  };

  class RealValue : public Value<double>
  {
  public:
    RealValue() : Value<double>() {}
    RealValue(const double& val) : Value<double>(val) {}
    RealValue(const RealValue& o) : Value<double>(o) {}

    inline void log10();
    inline void exp10();
    inline void abs();
    inline void atanRad();
    inline void atanDeg();
  };

  typedef Value<Complex> ComplexValue;

  inline void RealValue::log10()
  {
    if(isValid()) {
      setValue(::log10(value()));
    }
  }

  inline void RealValue::exp10()
  {
    if(isValid()) {
      setValue(::pow(10.0, value()));
    }
  }

  inline void RealValue::abs()
  {
    if(isValid()) {
      setValue(::fabs(value()));
    }
  }

  inline void RealValue::atanRad()
  {
    if(isValid()) {
      setValue(::atan(value()));
    }
  }

  inline void RealValue::atanDeg()
  {
    if(isValid()) {
      setValue(Angle::radiansToDegrees(::atan(value())));
    }
  }

} // namespace QGpCoreMath

#endif // VALUE_H
