/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-11-06
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "TriangularMatrix.h"
#include "Random.h"

namespace QGpCoreMath {

  /*!
    \class TriangularMatrix TriangularMatrix.h
    \brief A compact storage for symetric matrices

    Full description of class still missing
  */

  /*!
    \a n is the dimension of the matrix. For instance for n=5, a 5x5 matrix,
    we have to store 4+3+2+1=10 values.
  */
  TriangularMatrix::TriangularMatrix(int n)
  {
    _n=n;
    _n1=_n-1;
    _values=new double[valueCount()];
    setRows();
  }

  TriangularMatrix::TriangularMatrix(const TriangularMatrix& o)
  {
    _n=o._n;
    _n1=o._n1;
    int nval=valueCount();
    _values=new double[nval];
    memcpy(_values, o._values, nval*sizeof(double));
    setRows();
  }

  /*!
    Description of destructor still missing
  */
  TriangularMatrix::~TriangularMatrix()
  {
    delete [] _values;
    delete [] _rows;
  }

  void TriangularMatrix::operator=(const TriangularMatrix& o)
  {
    delete [] _values;
    delete [] _rows;
    _n=o._n;
    _n1=o._n1;
    int nval=valueCount();
    _values=new double[nval];
    memcpy(_values, o._values, nval*sizeof(double));
    setRows();
  }

  void TriangularMatrix::setRows()
  {
    _rows=new double *[_n1];
    double * p=_values-1;  // Add a shift to access columns with natural index (no diagonal)
    for(int i=0; i<_n1; i++) {
      _rows[i]=p-i;        // Add a shift to access columns with natural index (lower triangle)
      p+=columnCount(i);
    }
  }

  void TriangularMatrix::removeAt(int index)
  {
    int n=_n1-1;
    int i;
    double * rowValues;
    for(i=0; i<index; i++) {
      rowValues=_rows[i];
      for(int j=index; j<_n1; j++) {
        rowValues[j]=rowValues[j+1];
      }
    }
    for(; i<n; i++) {
      _rows[i]=_rows[i+1]+1;
    }
    _n--;
    _n1--;
  }

  void TriangularMatrix::random(double min, double max)
  {
    Random r;
    for(int i=valueCount()-1; i>=0;i--) {
      _values[i]=r.uniform(min, max);
    }
  }

  QString TriangularMatrix::toUserString(char format, int precision) const
  {
    QString str("    | ");
    for(int j=0; j<_n; j++) {
      str+=QString(" %1 |").arg(j, 12, 10, QChar(' '));
    }
    str+="\n----|-";
    for(int j=0; j<_n; j++) {
      str+="--------------|";
    }
    str+="\n";
    int j;
    for(int i=0; i<_n; i++) {
      str+=QString("%1 | ").arg(i, 3, 10, QChar(' '));
      for(j=0; j<=i; j++) {
        str+=QString("              |");
      }
      for(; j<_n; j++) {
        str+=QString(" %1 |").arg(Number::toDouble(constAt(i, j)), 12, format, precision, QChar(' '));
      }
      str+="\n";
    }
    str+="----|-";
    for(int j=0; j<_n; j++) {
      str+="--------------|";
    }
    str+="\n";
    return str;
  }

  QString TriangularMatrix::toOctaveString(char format, int precision) const
  {
    QString str("[");
    for(int i=0; i<_n; i++) {
      for(int j=0; j<_n; j++) {
        str+=QString(" %1").arg(Number::toString(constAt(i, j), format, precision));
      }
      str+=";";
    }
    str+=" ]";
    return str;
  }

  QString TriangularMatrix::toString() const
  {
    QString str;
    int n=valueCount();
    for(int i=0; i<n; i++) {
      str+=Number::toString(_values[i], 'g', 20);
      str+=" ";
    }
    return str;
  }

  QString TriangularMatrix::toGridString() const
  {
    QString str;
    str+="x y val\n";
    for(int i=0; i<_n; i++) {
      for(int j=0; j<_n; j++) {
        str+=QString::number(i);
        str+=" ";
        str+=QString::number(j);
        str+=" ";
        str+=Number::toString(Number::toDouble(constAt(j, i)), 'g', 20);
        str+="\n";
      }
    }
    return str;
  }

  bool TriangularMatrix::fromString(const StringSection& s)
  {
    const QChar * ptr=nullptr;
    int n=valueCount();
    for(int i=0; i<n; i++) {
      if(!s.nextNumber(_values[i], ptr)) {
        return false;
      }
    }
    return true;
  }

} // namespace QGpCoreMath

