/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-01-12
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Triangle.h"
#include "Rect.h"

namespace QGpCoreMath {

  /*!
    \class Triangle Triangle.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Source: https://en.wikipedia.org/wiki/Circumscribed_circle#Higher_dimensions

    The radius of the returned circle is not a distance but a squared distance.
    It avoids useless sqrt.
  */
  Circle Triangle::circumcircle(bool& ok) const
  {
    Point a=_vertices[0]-_vertices[2];
    Point b=_vertices[1]-_vertices[2];
    Point ab, p0, amb;

    amb=a-b;
    ab.vectorialProduct(a, b);

    double a2=a.abs2();
    double b2=b.abs2();
    double ab2=ab.abs2();
    if(ab2==0.0) {
      ok=false;
      return Circle(Point2D(), 0.0);
    }
    double invab2=0.5/ab2;
    b*=a2;
    a*=b2;
    b-=a;
    p0.vectorialProduct(b, ab);
    p0*=invab2;
    p0+=_vertices[2];
    // Due to precision issues, the theoretical radius computed this way
    // double r0=sqrt(0.5*a2*b2*invab2*amb.abs2());
    // can above or below all distances to vertices. All situations have been
    // encountered. Hence it is faster to compute the distance to all vertices
    // and keep the largest one.
    double rv0=p0.squareDistanceTo(_vertices[0]);
    double rv1=p0.squareDistanceTo(_vertices[1]);
    double rv2=p0.squareDistanceTo(_vertices[2]);
    double r=rv0;
    if(rv1>r) {
      r=rv1;
    }
    if(rv2>r) {
      r=rv2;
    }
    return Circle(p0, r);
  }

  /*
  // Testing of circumcircle()... distances to vertices must be null
  Point2D va(1, 1);
  Point2D vb(2, 1);
  Point2D vc(3, 3);
  Triangle t(va, vb, vc);
  bool ok=true;
  Circle c=t.circumcircle(ok);
  qDebug() << c.center().toString() << c.radius()
           << c.distanceTo(va)
           << c.distanceTo(vb)
           << c.distanceTo(vc);
  */

  /*!
    Init a super triangle containing all points
  */
  void Triangle::super(const QList<Point2D>& points)
  {
    // Get a box enclosing all points
    Rect r;
    QList<Point2D>::const_iterator it=points.begin();
    if(it==points.end()) {
      return;
    }
    r.setLimits(*it, *it);
    for(QList<Point2D>::const_iterator it=points.begin(); it!=points.end(); it++) {
      r.add(*it);
    }
    r.enlarge(0.1, LinearScale, LinearScale); // Avoid precision arrors in Delaunay triangulation
    _vertices[0].setX(r.x1()-r.width());
    _vertices[0].setY(r.y1());
    _vertices[1].setX(r.x2()+r.width());
    _vertices[1].setY(r.y1());
    _vertices[2].setX(r.x1()+0.5*r.width());
    _vertices[2].setY(r.y2()+0.5*r.height());
  }

  bool Triangle::hasEdge(const Point2D& p1, const Point2D& p2) const
  {
    if(_vertices[0]==p1) {
      return _vertices[1]==p2 || _vertices[2]==p2;
    }
    if(_vertices[1]==p1) {
      return _vertices[0]==p2 || _vertices[2]==p2;
    }
    if(_vertices[2]==p1) {
      return _vertices[0]==p2 || _vertices[1]==p2;
    }
    return false;
  }

  bool Triangle::hasVertex(const Point2D& p) const
  {
    return _vertices[0]==p || _vertices[1]==p || _vertices[2]==p;
  }

  bool Triangle::hasSimilarVertex(const Point2D& p, double tolerance) const
  {
    return _vertices[0].isSimilar(p, tolerance) ||
           _vertices[1].isSimilar(p, tolerance) ||
           _vertices[2].isSimilar(p, tolerance);
  }

  bool Triangle::hasSimilarEdge(const Point2D& p1, const Point2D& p2, double tolerance) const
  {
    if(_vertices[0].isSimilar(p1, tolerance)) {
      return _vertices[1].isSimilar(p2, tolerance) ||
             _vertices[2].isSimilar(p2, tolerance);
    }
    if(_vertices[1].isSimilar(p1, tolerance)) {
      return _vertices[0].isSimilar(p2, tolerance) ||
            _vertices[2].isSimilar(p2, tolerance);
    }
    if(_vertices[2].isSimilar(p1, tolerance)) {
      return _vertices[0].isSimilar(p2, tolerance) ||
             _vertices[1].isSimilar(p2, tolerance);
    }
    return false;
  }

  bool Triangle::isNeighbor(const Triangle& o) const
  {
    if(o.hasVertex(_vertices[0])) {
      return o.hasVertex(_vertices[1]) ||
             o.hasVertex(_vertices[2]);
    } else {
      return o.hasVertex(_vertices[1]) &&
             o.hasVertex(_vertices[2]);
    }
  }

  double Triangle::surface() const
  {
    Point2D ab=_vertices[1]-_vertices[0];
    Point2D ac=_vertices[2]-_vertices[0];
    return 0.5*fabs(ab.x()*ac.y()-ac.x()*ab.y());
  }

  QString Triangle::toString(int precision, char format) const
  {
    QString s;
    s+="[";
    s+=_vertices[0].toString(precision, format);
    s+="] [";
    s+=_vertices[1].toString(precision, format);
    s+="] [";
    s+=_vertices[2].toString(precision, format);
    s+="]";
    return s;
  }

} // namespace QGpCoreMath

