/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-10-13
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>
#include <QGpCoreTools.h>

#include "Stationarity.h"

namespace QGpCoreMath {

/*!
  \class Stationarity Stationarity.h
  \brief Monitors a statistical process

  Monitor a process (based on one value) for its stabilization.
  relativeAtEnd() or absoluteAtEnd() returns true if the maximum deviation
  does not exceed a ratio times the mean(), or simply a threshold.

  ratio is given in the constructor. The monitored length must be fixed in the
  constructor.
*/

/*!
  Sets queue length for stationarity window
*/
Stationarity::Stationarity(int queueLength, double ratio)
{
  TRACE;
  _queueLength=queueLength;
  _mean=0;
  _currentI=-1;
  _values=new double [_queueLength];
  _count=0;
  _imax=0;
  _maxdelta=std::numeric_limits<double>::infinity();
  _ratio=ratio;
}

Stationarity::~Stationarity()
{
  TRACE;
  delete [] _values;
}

void Stationarity::add(double newval)
{
  TRACE;
  if(_count<_queueLength) {
    _count++;
    double w=1.0/count();
    _mean=_mean*((double)(_count-1)*w)+newval*w;
  } else {
    _count++;
    _mean=_mean+(newval-_values[oldestIndex()])/_queueLength;
  }
  _values[nextIndex()]=newval;
}

/*
  Relative criteria
*/
bool Stationarity::relativeAtEnd()
{
  TRACE;
  if(_count<_queueLength) return false;
  if(_count>_imax+_queueLength) {
    _maxdelta=-std::numeric_limits<double>::infinity();
    for(int i=0;i<_queueLength;i++) {
      double delta=fabs(_mean-_values[i]);
      if(delta>_maxdelta) {
        _imax=_count-_queueLength+i;
        _maxdelta=delta;
      }
      if(_maxdelta>fabs(_ratio*_mean)) return false;
    }
    return true;
  } else {
    return _maxdelta<=fabs(_ratio*_mean);
  }
}

/*!
  Absolute criteria (_ratio is used as a threshold)
*/
bool Stationarity::absoluteAtEnd()
{
  TRACE;
  if(_count<_queueLength) return false;
  if(_count>_imax+_queueLength) {
    _maxdelta=-std::numeric_limits<double>::infinity();
    for(int i=0;i<_queueLength;i++) {
      double delta=fabs(_mean-_values[i]);
      if(delta>_maxdelta) {
        _imax=_count-_queueLength+i;
        _maxdelta=delta;
      }
      if(_maxdelta>_ratio) return false;
    }
    return true;
  } else {
    return _maxdelta<=_ratio;
  }
}

} // namespace QGpCoreMath
