/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-02
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef COLORMAPDATA_H
#define COLORMAPDATA_H

#include <QGpCoreTools.h>

#include "QGpCoreMathDLLExport.h"
#include "ColorPalette.h"
#include "SamplingParameters.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT ColorMapData : public QSharedData
  {
  public:
    ColorMapData();
    ColorMapData(const ColorMapData& o);
    ~ColorMapData();

    bool operator==(const ColorMapData& o) const;
    bool operator!=(const ColorMapData& o) const {return ! operator==(o);}
    bool operator<(const ColorMapData& o) const;

    ColorPalette palette() const {return _palette;}
    void setPalette(const ColorPalette& pal);
    inline void generateColorScale(int n, ColorPalette::Model m=ColorPalette::McNamesClip,
                                   bool reversed=false, int transparency=255);
    inline void generateGrayScale(int n, ColorPalette::Model m=ColorPalette::McNamesClip,
                                  bool reversed=false, int transparency=255);
    void rgbInterpole(int imin, int imax) {_palette.rgbInterpole(imin, imax);}
    void hsvInterpole(int imin, int imax) {_palette.hsvInterpole(imin, imax);}
    void setWhiteTransparent(bool b) {_palette.setWhiteTransparent(b);}

    void setValues(double min, double max, SamplingParameters::ScaleType t) {setValues(0, count()-2, min, max, t);}
    void setValues(int imin, int imax, double min, double max, SamplingParameters::ScaleType t);

    void resize(int n);
    int count() const {return _palette.count();}

    void setUpperValue(int i, double v) {_upperValues[i]=v;}
    double upperValue(int i) const {return i<count()-1 ? _upperValues[i] : std::numeric_limits<double>::infinity();}
    double lowerValue(int i) const {return i>0 ? _upperValues[i-1] : -std::numeric_limits<double>::infinity();}

    const Color& color(int index) const {return _palette.color(index);}
    Color& color(int index) {return _palette.color(index);}
    const Color& color(double val) const {return _palette.color(index(val));}
    Color& color(double val) {return _palette.color(index(val));}
    void setColor(int index, const Color& c) {_palette.setColor(index, c);}

    int index(double val) const;
  private:
    ColorPalette _palette;
    double * _upperValues;
    int _n2;
  };

  inline void ColorMapData::generateColorScale(int n, ColorPalette::Model m,
                                               bool reversed, int transparency)
  {
    resize(n);
    _palette.generateColorScale(n, m, reversed, transparency);
  }

  inline void ColorMapData::generateGrayScale(int n, ColorPalette::Model m,
                                              bool reversed, int transparency)
  {
    resize(n);
    _palette.generateGrayScale(n, m, reversed, transparency);
  }

} // namespace QGpCoreMath

#endif // COLORMAPDATA_H
