/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-01-26
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ChebyshevFilter.h"

namespace QGpCoreMath {

  /*!
    \class ChebyshevFilter ChebyshevFilter.h
    \brief A Chebyshev filter


    T_n(cos theta)=cos(n theta)
  */

  /*!

  */
  ChebyshevFilter::ChebyshevFilter(int order, double rippleFactor)
  {
    _order=order;
    _ripple=rippleFactor;
  }

  /*!
    Designs a ChebyshevFilter filter.
    \a wPass1 and \a wPass2 define the pass band.
    \a wStop1 and \a wStop2 define the stop band.
    If the pass band is contained within the stop band, the filter is a band-pass filter.
    If the stop band is contained within the pass band, the filter is a band-stop filter.
  */
  ChebyshevFilter::ChebyshevFilter(double wPass1, double wPass2, double wStop1, double wStop2,
                                   double allowedPassRipple, double allowedStopRipple)
  {

  }

  /*!
    Designs a ChebyshevFilter filter.
    \a wPass is the pass band cut-off.
    \a wStop is the stop band cut-off.
    If \a wPass is lower than \a wStop, the filter is a low-pass filter.
    If \a wStop is lower than \a wPass, the filter is a high pass filter.
  */
  ChebyshevFilter::ChebyshevFilter(double wPass, double wStop,
                                   double allowedPassRipple, double allowedStopRipple)
  {

  }

  /*!
    Constructed from the recurrence relation
    Source: https://en.wikipedia.org/wiki/Chebyshev_polynomials#Definition
  */
  double ChebyshevFilter::polynomial(double x) const
  {
    int n=_order+1;
    double * values=new double[n<2 ? 2 : n];
    values[0]=1.0;
    values[1]=x;
    double x2=2.0*x;
    for(int i=2; i<=_order; i++) {
      values[i]=x2*values[i-1]-values[i-2];
    }
    x=values[_order];
    delete [] values;
    return x;
  }

  /*!
    Type I Chebyshev filters

    \a wR is the relative angular frequency w/w0, where w0 is the cut-off frequency.
  */
  double ChebyshevFilter::value(double wR) const
  {
    double t=polynomial(wR);
    t*=_ripple;
    return 1.0/sqrt(1.0+t*t);
  }

} // namespace QGpCoreMath

