/***************************************************************************
**
**  This file is part of HVCore.
**
**  HVCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-11
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "HVStationSignals.h"

/*!
  \class HVStationSignals HVStationSignals.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

namespace HVCore {

  HVStationSignals::~HVStationSignals()
  {
    TRACE;
  }

  bool HVStationSignals::setProcessed(const StationProcessSignals * from,
                                      const TimeRange& tw, const AbstractParameters * param)
  {
    TRACE;
    const HVParameters * hvParam=static_cast<const HVParameters *>(param);
    for(int i=0; i<nComponents(); i++) {
      if(!fft(from, i, tw, hvParam)) {
        return false;
      }
    }
    // Smoothed amplitude for vertical component
    _verticalSpectrum=processed(0)->spectrumAmplitudeCurve();
    _verticalSpectrum.smooth(hvParam->frequencySampling(), hvParam->smoothing());
    return true;
  }

  Curve<Point2D> HVStationSignals::horizontal(const HVParameters * param, double factor)
  {
    TRACE;
    Curve<Point2D> h=processed(1)->spectrumAmplitudeCurve();
    h.ySquare();
    Curve<Point2D> h2=processed(2)->spectrumAmplitudeCurve();
    h2.ySquare();
    h.ySum(h2);
    if(factor!=1.0) {
      h.yMultiply(factor);
    }
    h.ySqrt();
    h.smooth(param->frequencySampling(), param->smoothing());
    return h;
  }

  Curve<Point2D> HVStationSignals::horizontalGeometric(const HVParameters * param)
  {
    TRACE;
    Curve<Point2D> h=processed(1)->spectrumAmplitudeCurve();
    Curve<Point2D> h2=processed(2)->spectrumAmplitudeCurve();
    h.yMultiply(h2);
    h.ySqrt();
    h.smooth(param->frequencySampling(), param->smoothing());
    return h;
  }

  /*!
    Return an horizontal spectrum in direction \a azimuth counted clockwise from North
  */
  Curve<Point2D> HVStationSignals::horizontal(double azimuth, const HVParameters *param)
  {
    TRACE;
    double a=Angle::degreesToRadians(azimuth);
    ComplexSignal * h1c=processed(1)->positiveComplexSpectrum();
    h1c->multiply(cos(a));
    ComplexSignal * h2c=processed(2)->positiveComplexSpectrum();
    h2c->multiply(sin(a));
    h1c->add(h2c);
    Curve<Point2D> h=h1c->abs(1.0/processed(1)->duration());
    h.smooth(param->frequencySampling(), param->smoothing());
    return h;
  }

  bool HVStationSignals::fft(const StationProcessSignals * from, int iComp,
                             const TimeRange& tw, const HVParameters *param)
  {
    TRACE;
    if(!copyOriginalSignal(iComp, tw, from)) {
      return false;
    }
    DoubleSignal *& psig=processed(iComp);
    psig->subtractValue();
    psig->taper(param->tapering());
    psig->fastFourierTransform(DoubleSignal::Spectrum);
    return true;
  }

} // namespace HVCore
