/***************************************************************************
**
**  This file is part of GeopsySLink.
**
**  GeopsySLink is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsySLink is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-04-23
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SEEDLINK_H
#define SEEDLINK_H

#include <GeopsyCore.h>

#include "GeopsySLinkDLLExport.h"
#include "SeedLinkServer.h"
#include "SeedLinkStream.h"

namespace GeopsySLink {

class SeedLinkPrivate;

class GEOPSYSLINK_EXPORT SeedLink : public Thread
{
  Q_OBJECT
public:
  SeedLink(QObject * parent=nullptr);
  ~SeedLink();

  void setServer(QByteArray address, qint16 port);
  bool hasServer() const {return !_serverAddress.isEmpty();}

  QString serverName() const {return _serverName;}
  quint16 serverPort() const {return _serverPort;}
  const SeedLinkServer& serverInfo() const {return _serverInfo;}

  void start();
  void stop();
  void streams();
  void stations();
  void addStation(SeedLinkStation * station) {_serverInfo.addStation(station);}
  bool addStream(SeedLinkStream * str);
  bool removeStream(SeedLinkStream * str);

  int listeningStreamCount() {return _listeningStreams.count();}
  SubSignalPool subPool() const;

  void setFromTime(const DateTime& f) {_fromTime=f;}
  void setBufferType(SeedLinkStream::BufferType bt) {_bufferType=bt;}
  void setMaximumDuration(double d) {_maximumDuration=d;}
  void setCurrentDatabase(SignalDatabase * db) {_serverInfo.setDatabase(db);}

  enum Error {NoError, StationsNotAvailable, StreamsNotAvailable, FromTimeNotAvailable};
  Error error() const {return _error;}
protected:
  virtual void run();
signals:
  void infoAvailable();
  void beginSignalChange(Signal * sig, TimeRange tw);
  void endSignalChange(Signal * sig, TimeRange tw);
  void error(GeopsySLink::SeedLink::Error e);
private:
  static SeedLink * _instance;
  static void log(const char * msg);
  void packetReceived();
  void setError(Error e);

  enum Request {Streams, Stations, Data, Dns};
  Request _request;
  Mutex _mutex;
  Error _error;

  QByteArray _serverName;
  qint16 _serverPort;
  QByteArray _serverAddress;
  SeedLinkPrivate * _internals;

  QString _xmlInfos;
  SeedLinkServer _serverInfo;

  QMap<QByteArray,SeedLinkStream *> _listeningStreams;

  DateTime _fromTime;
  double _maximumDuration;
  SeedLinkStream::BufferType _bufferType;
};

} // namespace GeopsySLink

// Allow passing SeedLink::Error as a QVariant or through SIGNAL/SLOT
Q_DECLARE_METATYPE(GeopsySLink::SeedLink::Error)

#endif // SEEDLINK_H
