/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-11-10
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyCore.h>
#include <QGpCoreTools.h>
#include <QGpGuiTools.h>

#include "SubPoolWindow.h"
#include "AbstractToolWidget.h"
#include "HeaderWidget.h"
#include "GeopsyGuiEngine.h"

namespace GeopsyGui {

SubPoolWindow::SubPoolWindow(QWidget* parent)
    : TitledWidget(parent)
{
  TRACE;
  _tool=nullptr;
  _headerWidget=nullptr;
  _signalProcess=nullptr;
  _locked=false;
  setAttribute(Qt::WA_DeleteOnClose);
}

SubPoolWindow::~SubPoolWindow ()
{
  TRACE;
  delete _signalProcess;
}

void SubPoolWindow::setSubPool(const SubSignalPool& subPool)
{
  TRACE;
  _subPool=subPool;
  subPoolUpdate();
}

void SubPoolWindow::setHeaderWidget(HeaderWidget * w)
{
  TRACE;
  _headerWidget=w;
  if(_headerWidget) {
    connect(_headerWidget, SIGNAL(destroyed()), this, SLOT(setHeaderWidget()));
  }
}

void SubPoolWindow::closeEvent(QCloseEvent * e)
{
  TRACE;
  if(isLocked()) {
    e->ignore();
    return;
  }
  if(_tool) {
    _tool->closeChildren();
    Settings::setSize(_tool->parentWidget(), _tool->objectName());
    GeopsyGuiEngine::instance()->removeSubWindow(_tool);
    _tool=nullptr;
  }
  if(_headerWidget) {
    GeopsyGuiEngine::instance()->removeSubWindow(_headerWidget);
    _headerWidget=nullptr;
  }
  e->accept();
}

void SubPoolWindow::setWindowTitle(QString t)
{
  TRACE;
  if(t.length()>100) {
    t.truncate(100);
    t+="...";
  }
#if(QT_VERSION > QT_VERSION_CHECK(5, 0, 0))
  QWidget::setWindowTitle(t);
#else
  TitledWidget::setWindowTitle(t);
#endif
}

void SubPoolWindow::addSignals(const SubSignalPool& sel)
{
  TRACE;
  if(_tool) {
    Message::warning(MSG_ID,tr("Adding signals"),
                         tr("The added signals will not be processed as the others. "
                         "To process all of them create a new window and start your "
                         "interpretation tool."), Message::ok());
  }
  if(_subPool.isEmpty())
    _subPool=sel;
  else
    _subPool.addSubPool(sel);
  subPoolUpdate();
}

/*!
  \fn subPoolUpdate()
  Function to redefine in child classes to update view after SubPool
  changes.
*/

/*!
  Function to redefine in child class to update view before changes
  to waveforms. Do not forget to call this function on exit.
*/
void SubPoolWindow::beginSignalChange(Signal * sig)
{
  if(_tool) {
    _tool->beginSignalChange(sig);
  }
}

/*!
  Function to redefine in child class to update view after changes
  to waveforms. Do not forget to call this function on exit.

  The default implementation send signal signalTypeChanged() with argument
  set to UndefinedSignalType (as if the signals of the SubPool have not the same type).
*/
void SubPoolWindow::endSignalChange(Signal * sig)
{
  TRACE;
  if(_tool) {
    _tool->endSignalChange(sig);
  }
  DoubleSignal::SignalType st=DoubleSignal::UndefinedSignalType;
  emit signalTypeChanged(st);
}

void SubPoolWindow::addFile(SignalDatabase * db, QString fileName)
{
  TRACE;
  SignalFile * f=db->filePool().find(fileName);
  if(f) {
    SubSignalPool sp;
    sp.addFile(f);
    addSignals(sp);
  }
}

void SubPoolWindow::addSignal(SignalDatabase * db, int id)
{
  TRACE;
  SubSignalPool sp;
  sp.addSignal(db, id);
  addSignals(sp);
}

void SubPoolWindow::addSignal(SubPoolWindow * viewer, int index)
{
  TRACE;
  if(!viewer) return;
  const SubSignalPool& sp=viewer->subPool();
  if(index >= sp.count()) return;
  SubSignalPool sel;
  sel.addSignal(sp.at(index) );
  addSignals(sp);
}

/*!
  Prevent subpool parent widget to be closed
*/
void SubPoolWindow::lock()
{
  TRACE;
  _locked=true;
}

/*!
  Allows subpool parent widget to be closed
*/
void SubPoolWindow::unlock()
{
  TRACE;
  _locked=false;
}

/*!
  Returns true if locked (prevent parent widget to be closed)
*/
bool SubPoolWindow::isLocked()
{
  TRACE;
  return _locked;
}

/*!
  Returns signal process object attached to this subpool (create a new one if not yet created).
*/
SignalProcess * SubPoolWindow::signalProcess()
{
  if(!_signalProcess) {
    _signalProcess=new SignalProcess;
    _signalProcess->setCurrentSubPool(&_subPool);
  }
  return _signalProcess;
}

} // namespace GeopsyGui
