/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-12-22
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "InstrumentalResponse.h"

namespace GeopsyCore {

  /*!
    \class InstrumentalResponse InstrumentalResponse.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  const QString InstrumentalResponse::xmlInstrumentalResponseTag="InstrumentalResponse";

  /*!
    Description of constructor still missing
  */
  InstrumentalResponse::InstrumentalResponse()
  {
    TRACE;
    _noiseLevel=1e-16;
    _transductionFactor=1.0;
    _normalizationFactor=1.0;
    _frequencyFactor=2.0*M_PI;
  }

  /*!
    Description of destructor still missing
  */
  InstrumentalResponse::~InstrumentalResponse()
  {
    TRACE;
  }

  /*!
    Deconvolution by the instrumental response defined by \a poles and \a zeros.
    Wiener filter included in the deconvolution (https://en.wikipedia.org/wiki/Wiener_deconvolution).
    White noise is assumed.
  */
  Complex InstrumentalResponse::deconvolution(double frequency, const Complex& sample) const
  {
    Complex hn(1.0, 0.0), hd(1.0, 0.0);
    Complex z(0.0, frequency*_frequencyFactor);
    for(QVector<Complex>::const_iterator it=_zeros.begin(); it!=_zeros.end(); it++) {
      hn*=(z-*it);
    }
    for(QVector<Complex>::const_iterator it=_poles.begin(); it!=_poles.end(); it++) {
      hd*=(z-*it);
    }
    if(hn.isNull()) {
      return sample;
    } else if(hd.isNull()){
      return Complex();
    } else {
      double h2=(hn/hd).abs2();
      double wienerFactor=h2/(h2+_noiseLevel/sample.abs());
      return sample*(hd/hn)*(wienerFactor*_normalizationFactor);
    }
  }

  Complex InstrumentalResponse::value(double frequency) const
  {
    Complex hn(1.0, 0.0), hd(1.0, 0.0);
    Complex z(0.0, frequency*_frequencyFactor);
    for(QVector<Complex>::const_iterator it=_zeros.begin(); it!=_zeros.end(); it++) {
      hn*=(z-*it);
    }
    for(QVector<Complex>::const_iterator it=_poles.begin(); it!=_poles.end(); it++) {
      hd*=(z-*it);
    }
    if(hn.isNull()) {
      return Complex(_transductionFactor*_normalizationFactor);
    } else if(hd.isNull()){
      return Complex();
    } else {
      return (hn/hd)*(_transductionFactor*_normalizationFactor);
    }
  }

  QString InstrumentalResponse::arguments() const
  {
    TRACE;
    QString s;
    s+=QString::number(_poles.count());
    s+=", ";
    for(QVector<Complex>::const_iterator it=_poles.begin(); it!=_poles.end(); it++) {
      s+=QString::number(it->re());
      s+=", ";
      s+=QString::number(it->im());
      s+=", ";
    }
    s+=QString::number(_zeros.count());
    s+=", ";
    for(QVector<Complex>::const_iterator it=_zeros.begin(); it!=_zeros.end(); it++) {
      s+=QString::number(it->re());
      s+=", ";
      s+=QString::number(it->im());
      s+=", ";
    }
    s+=QString::number(_noiseLevel);
    return s;
  }

  void InstrumentalResponse::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    Q_UNUSED(context)
    writeProperty(s, "name", _name);
    writeProperty(s, "transductionFactor", _transductionFactor);
    writeProperty(s, "normalizationFactor", _normalizationFactor);
    writeProperty(s, "frequencyFactor", _frequencyFactor);
    writeProperty(s, "noiseLevel", _noiseLevel);
    for(QVector<Complex>::const_iterator it=_poles.begin(); it!=_poles.end(); it++) {
      writeProperty(s, "pole", it->toString('g', 20));
    }
    for(QVector<Complex>::const_iterator it=_zeros.begin(); it!=_zeros.end(); it++) {
      writeProperty(s, "zero", it->toString('g', 20));
    }
  }

  XMLMember InstrumentalResponse::xml_member(XML_MEMBER_ARGS)
  {
    Q_UNUSED(context)
    Q_UNUSED(attributes)
    if(tag=="pole") return XMLMember(3);
    else if(tag=="zero") return XMLMember(4);
    else if(tag=="name") return XMLMember(0);
    else if(tag=="transductionFactor") return XMLMember(1);
    else if(tag=="normalizationFactor") return XMLMember(5);
    else if(tag=="frequencyFactor") return XMLMember(6);
    else if(tag=="noiseLevel") return XMLMember(2);
    return XMLMember(XMLMember::Unknown);
  }

  bool InstrumentalResponse::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    Q_UNUSED(context)
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    switch(memberID) {
    case 0:
      _name=content.toString();
      return true;
    case 1:
      _transductionFactor=content.toDouble();
      return true;
    case 2:
      _noiseLevel=content.toDouble();
      return true;
    case 3:
      addPole(content.toComplex());
      return true;
    case 4:
      addZero(content.toComplex());
      return true;
    case 5:
      _normalizationFactor=content.toDouble();
      return true;
    case 6:
      _frequencyFactor=content.toDouble();
      return true;
    }
    return false;
  }

} // namespace GeopsyCore

