/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-11-10
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SPACPeaks.h"
#include "SPACParameters.h"
#include "ArraySelection.h"

namespace ArrayCore {

const char * SPACPeaks::maxHeaderLine="# abs_time frequency polarization ring autocorr valid";

  /*!
    \class SPACPeaks SPACPeaks.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SPACPeaks::SPACPeaks()
    : IncreaseStorage(1024)
  {
    TRACE;
    _values=static_cast<Value *>(allocateVector(sizeof(Value)));
    _full=false;
  }

  /*!
    Description of destructor still missing
  */
  SPACPeaks::~SPACPeaks()
  {
    TRACE;
    free(_values);
  }

  void SPACPeaks::clear()
  {
    TRACE;
    IncreaseStorage::clear();
    free(_values);
    _values=static_cast<Value *>(allocateVector(sizeof(Value)));
    _full=false;
  }


  void SPACPeaks::reallocate()
  {
    TRACE;
    double n=capacity();
    n*=sizeof(Value);
    n/=1024.0*1024.0;
    _dataLock.lockForWrite();
    Value * newValues=static_cast<Value *>(reallocateVector((uchar *)_values, sizeof(Value)));
    if(newValues) {
      _values=newValues;
    } else {
      App::log(tr("Failed to reallocate results for %1 values (size=%2 Mb)\n"
                  "Container is full.")
               .arg(capacity())
               .arg(n));
      _full=true;
    }
    _dataLock.unlock();
    APP_LOG(1, tr("Reallocated results for %1 values (size=%2 Mb)\n")
            .arg(capacity())
            .arg(n))
  }

  bool SPACPeaks::add(Mode::Polarization polarization, const SPACResults::ValueInfo& info,
                      int ring, double autocorr)
  {
    if(_full) {
      return false;
    }
    int n=size();
    IncreaseStorage::add();
    Value& v=_values[n];
    v.time=_timeReference.secondsTo(info.time);
    v.frequency=info.frequency;
    v.ring=ring;
    v.autocorr=autocorr;
    v.polarization=polarization;
    v.valid=true;
    return true;
  }


  void SPACPeaks::add(const SPACPeaks& peaks)
  {
    App::log(tr("Merging results...\n"));
    _addLock.lock();
    int i0=size();
    int n=peaks.size();
    IncreaseStorage::add(n);
    _dataLock.lockForRead();
    _addLock.unlock();
    for(int i=0; i<n; i++) {
      _values[i0+i]=peaks._values[i];
    }
    _dataLock.unlock();
  }

  /*!
    Sort as primary key decreasing frequency and secondary increasing time
  */
  void SPACPeaks::sort()
  {
    TRACE;
    int n=size();
    QVector<Value *> list(n);
    for(int i=0; i<n; i++) {
      list[i]=_values+i;
    }
    std::sort(list.begin(), list.end(), valueLessThan);
    Value * newValues=static_cast<Value *>(allocateVector(sizeof(Value)));
    for(int i=0; i<n; i++) {
      memcpy(newValues+i, list.at(i), sizeof(Value));
    }
    free(_values);
    _values=newValues;
  }

  bool SPACPeaks::valueLessThan(Value * v1, Value * v2)
  {
    if(v1->frequency>v2->frequency) {
      return true;
    } else if(v1->frequency<v2->frequency) {
      return false;
    } else {
      if(v1->time<v2->time) {
        return true;
      } else if(v1->time>v2->time) {
        return false;
      } else {
        if(v1->polarization<v2->polarization) {
          return true;
        } else if(v1->polarization>v2->polarization) {
          return false;
        } else {
          if(v1->ring<v2->ring) {
            return true;
          } else if(v1->ring>v2->ring) {
            return false;
          } else {
            return v1->autocorr<v2->autocorr;
          }
        }
      }
    }
  }

  bool SPACPeaks::save(const QString &fileName,
                       const SPACParameters& param,
                       const ArraySelection& array,
                       const QString& log) const
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      s.setRealNumberPrecision(20);
      s << "# MAX FORMAT RELEASE 1.1\n";
      CoreApplication::signature(s);
      s << "#\n"
           "# BEGIN STATION LIST\n"
           "#\n";
      s << array.toString(ArraySelection::Absolute);
      s << "#\n"
           "# END STATION LIST\n"
           "#\n"
           "#\n"
           "# BEGIN PARAMETERS\n"
           "#\n";
      s << param.toString();
      s << "#\n"
           "# END PARAMETERS\n"
           "#\n"
           "#\n"
           "# BEGIN LOG\n"
           "#\n";
      s << log;
      s << "#\n"
           "# END LOG\n"
           "#\n"
           "#\n"
           "# BEGIN DATA\n"
           "#\n";
      s << maxHeaderLine << "\n";
      int n=size();
      for(int i=0; i<n; i++) {
        Value& v=_values[i];
        s << _timeReference.shifted(v.time).toString() << " "
          << v.frequency << " "
          << Mode::polarization(v.polarization) << " "
          << v.ring << " "
          << v.autocorr << " "
          << (v.valid ? 1 : 0) << "\n";
      }
      return true;
    } else {
      App::log(tr("error writing to file '%1'\n").arg(fileName) );
      return false;
    }
  }

} // namespace ArrayCore

