/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-08
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "FKParameters.h"

namespace ArrayCore {

  /*!
    \class FKParameters FKParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  FKParameters::FKParameters()
    : ArrayParameters()
  {
    setDefaultValues(RTBF, RefinedGrid);
    _cacheGridStepFactor=0.05;
    _cacheGridStep=0.0;
    _gridStep=0.0;
    _gridSize=0.0;
    _kmin=0.0;
    _kmax=0.0;
    _smin=1.0/3500.0;
    _smax=1.0/50.0;
    _minAzimuth=0.0;
    _maxAzimuth=0.0;
    _sourceGridStep=1.0;
    _sourceGridSize=0.0;
  }

  FKParameters::FKParameters(const FKParameters& o)
    : ArrayParameters(o)
  {
    _inversionMethod=o._inversionMethod;
    _cacheGridStepFactor=o._cacheGridStepFactor;
    _cacheGridStep=o._cacheGridStep;
    _gridStepFactor=o._gridStepFactor;
    _gridStep=o._gridStep;
    _gridSizeFactor=o._gridSizeFactor;
    _gridSize=o._gridSize;
    _kmin=o._kmin;
    _kmax=o._kmax;
    _smin=o._smin;
    _smax=o._smax;
    _minAzimuth=o._minAzimuth;
    _maxAzimuth=o._maxAzimuth;
    _refineSlowMin=o._refineSlowMin;
    _refineSlowMax=o._refineSlowMax;
    _maxPeakCount=o._maxPeakCount;
    _absoluteThreshold=o._absoluteThreshold;
    _relativeThreshold=o._relativeThreshold;
    _saveUndefinedEllipticities=o._saveUndefinedEllipticities;
    _exportAllFKGrids=o._exportAllFKGrids;
    _damping=o._damping;
    _rotateStepCount=o._rotateStepCount;
    _processType=o._processType;
    _arrays=o._arrays;
    _fixedEllipticityFileName=o._fixedEllipticityFileName;
    _minimumDistance=o._minimumDistance;
    _maximumDistance=o._maximumDistance;
    _sourceGridStep=o._sourceGridStep;
    _sourceGridSize=o._sourceGridSize;
  }

  void FKParameters::setDefaultValues(ProcessType t, InversionMethod invm)
  {
    _processType=t;
    _maxPeakCount=INT_MAX; // Avoid sensitivity to negligible differences between
                           // peak amplitudes. Accept all peaks down to
                           // 90% of the maximum amplitude rather than a single value
                           // If null ellipticities are not saved, max file size is kept
                           // reasonable even without filter (RTBF and RDS).
    _absoluteThreshold=0.0;
    _relativeThreshold=90.0;

    _saveUndefinedEllipticities=false;
    _exportAllFKGrids=false;
    _damping=0.0;
    _rotateStepCount=72;
    setfrequencyBandwidth(0.05);
    blockAveraging().setStatisticCount(50);
    blockAveraging().setStatisticMaxOverlap(0.0);

    switch(_processType) {
    FKPARAMETERS_PASSIVE_PROCESS_TYPES
      windowing().setPeriodCount(100.0);
      setOversamplingFactor(1.0);
      _minimumDistance=0.0;
      _maximumDistance=std::numeric_limits<double>::infinity();
      windowing().setSeismicEventTrigger(false);
      setSelectDurationFactor(0.0);
      break;
    FKPARAMETERS_ACTIVE_PROCESS_TYPES
      windowing().setLength(5.0);
      setOversamplingFactor(5.0);
      _minimumDistance=10.0;
      _maximumDistance=100.0;
      windowing().setSeismicEventTrigger(true);
      windowing().setSeismicEventDelay(-0.1);
      setSelectDurationFactor(0.0); // required for multi-shot with huge gaps
      break;
    }

    switch(_processType) {
    case RDS:
    case RDSRadial:
    case RDSVertical:
    case RDSRefined:
    case RDSSingle:
    case RDSFixedEll:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(6.0);
      _relativeThreshold=0.0; //
      break;
    case Capon:
    case CaponRadial:
    case CaponTransverse:
    case Omni:
    case PoggiRadial:
    case PoggiVertical:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(2.0);
      break;
    case RTBF:
    case RTBFRadial:
    case RTBFFixedEll:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(4.0);
      _relativeThreshold=0.0;
      invm=RefinedGrid; // Gradient not supported for RTBF
      break;
    case LDS:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(4.0);
      break;
    case Conventional:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
    case ActiveConventionalRayleigh:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(0.5);
      setMaximumPeakCount(1);
      break;
    case ActiveConventional:
      blockAveraging().setCount(1);
      setMaximumPeakCount(1);
      break;
    case ActiveRTBF:
    case ActiveDS:
    case ActiveCapon:
      blockAveraging().setCount(3);
      break;
    }
    _inversionMethod=invm;
    _gridStepFactor=defaultGridStepFactor(t, _inversionMethod);
    _gridSizeFactor=2.0;
    // Force default grid step and size if they are not yet set
    setKmin(_kmin);
    setKmax(_kmax);
  }

  double FKParameters::defaultGridStepFactor(ProcessType t, InversionMethod m)
  {
    switch(t) {
    case Capon:
    case Omni:
    case CaponRadial:
    case CaponTransverse:
    case RTBF:
    case RTBFRadial:
    case RTBFFixedEll:
    case RDS:
    case RDSRadial:
    case RDSVertical:
    case RDSRefined:
    case RDSSingle:
    case RDSFixedEll:
    case PoggiVertical:
    case PoggiRadial:
    case LDS:
    case ActiveCapon:
    case ActiveRTBF:
    case ActiveDS:
      switch(m) {
      case Gradient:
        return 0.5;
      case RefinedGrid:
        return 0.1;
      }
    case Conventional:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
    case ActiveConventional:
    case ActiveConventionalRayleigh:
      break;
    }
    switch(m) {
    case Gradient:
      break;
    case RefinedGrid:
      return 0.25;
    }
    return 1.0;
  }

  double FKParameters::cacheGridStep() const
  {
    if(_cacheGridStep==0.0) {
      return _kmin*_cacheGridStepFactor;
    } else {
      return _cacheGridStep;
    }
  }

  double FKParameters::gridStep() const
  {
    if(_gridStep==0.0) {
      return _kmin*_gridStepFactor;
    } else {
      return _gridStep;
    }
  }

  double FKParameters::gridSize() const
  {
    if(_gridSize==0.0) {
      return _kmax*_gridSizeFactor;
    } else {
      return _gridSize;
    }
  }

  void FKParameters::setKmin(double k)
  {
    _kmin=k;
  }

  void FKParameters::setKmax(double k)
  {
    _kmax=k;
  }

  /*!
    Limit grid size to fall inside the frequency-slowness area of interest.
  */
  double FKParameters::effectiveGridSize() const
  {
    double ksmax=2.0*M_PI*frequencySampling().maximum()*_smax;
    double gs=gridSize();
    // For the GUI frequency sampling maximum is usually null
    if(ksmax<gs && ksmax>0.0) {
      return ksmax;
    } else {
      return gs;
    }
  }

  /*!
    Convenience function to get the effective grid size at \a frequnecy,
    taking into account the maximum slowness.
  */
  double FKParameters::effectiveGridSize(double frequency) const
  {
    TRACE;
    double ksmax=2.0*M_PI*frequency*_smax;
    double gs=gridSize();
    if(ksmax<gs) {
      return ksmax;
    } else {
      return gs;
    }
  }

  void FKParameters::clearCurveRefine()
  {
    TRACE;
    _refineSlowMin.clear();
    _refineSlowMax.clear();
  }

  void FKParameters::addCurveRefine(double f, double minSlow, double maxSlow)
  {
    TRACE;
    _refineSlowMin.append(Point2D(f, minSlow));
    _refineSlowMax.append(Point2D(f, maxSlow));
  }

  int FKParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 29+ArrayParameters::keywordCount();
  }

  void FKParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=ArrayParameters::keywordCount();
    ArrayParameters::collectKeywords(keywords, prefix, suffix);
    keywords.add(prefix+"INVERSION_METHOD"+suffix, this, baseIndex+19);
    keywords.add(prefix+"CACHE_GRID_STEP"+suffix, this, baseIndex+20);
    keywords.add(prefix+"CACHE_GRID_STEP_FACTOR"+suffix, this, baseIndex+21);
    keywords.add(prefix+"GRID_STEP"+suffix, this, baseIndex+16);
    keywords.add(prefix+"GRID_STEP_FACTOR"+suffix, this, baseIndex+18);
    keywords.add(prefix+"GRID_SIZE"+suffix, this, baseIndex+17);
    keywords.add(prefix+"GRID_SIZE_FACTOR"+suffix, this, baseIndex+28);
    keywords.add(prefix+"K_MIN"+suffix, this, baseIndex);
    keywords.add(prefix+"K_MAX"+suffix, this, baseIndex+1);
    keywords.add(prefix+"MIN_V"+suffix, this, baseIndex+2);
    keywords.add(prefix+"MAX_V"+suffix, this, baseIndex+24);
    keywords.add(prefix+"MIN_AZIMUTH"+suffix, this, baseIndex+25);
    keywords.add(prefix+"MAX_AZIMUTH"+suffix, this, baseIndex+26);
    keywords.add(prefix+"N_MAXIMA"+suffix, this, baseIndex+3);
    keywords.add(prefix+"ABSOLUTE_THRESHOLD"+suffix, this, baseIndex+4);
    keywords.add(prefix+"RELATIVE_THRESHOLD"+suffix, this, baseIndex+5);
    keywords.add(prefix+"EXPORT_ALL_FK_GRIDS"+suffix, this, baseIndex+6);
    keywords.add(prefix+"SAVE_UNDEFINED_ELLIPTICITIES"+suffix, this, baseIndex+27);
    keywords.add(prefix+"DAMPING_FACTOR"+suffix, this, baseIndex+7);
    keywords.add(prefix+"ROTATE_STEP_COUNT"+suffix, this, baseIndex+8);
    keywords.add(prefix+"PROCESS_TYPE"+suffix, this, baseIndex+9);
    keywords.add(prefix+"ARRAY"+suffix, this, baseIndex+10);
    keywords.add(prefix+"SKIP_LOVE"+suffix, this, baseIndex+11);        // For compatibility
    keywords.add(prefix+"FIXED_ELLIPTICITY_FILE_NAME"+suffix, this, baseIndex+12);
    keywords.add(prefix+"MINIMUM_DISTANCE"+suffix, this, baseIndex+13);
    keywords.add(prefix+"MAXIMUM_DISTANCE"+suffix, this, baseIndex+14);
    keywords.add(prefix+"DO_DAMPING_FACTOR"+suffix, this, baseIndex+15); // For compatibility
    keywords.add(prefix+"SOURCE_GRID_STEP"+suffix, this, baseIndex+22);
    keywords.add(prefix+"SOURCE_GRID_SIZE"+suffix, this, baseIndex+23);
  }

  QString FKParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=ArrayParameters::toString(prefix, suffix);
    log+="#\n"
         "#\n"
         "#     FK method\n"
         "#\n"
         "#\n";
    log+="# Process types:\n"
         "#  [Comp is the required components]\n"
         "#  Keyword                  Comp Comments\n"
         "#  Conventional             Z    Conventional FK processing.\n"
         "#  ConventionalRadial       EN   Conventional FK processing for radial projections.\n"
         "#  ConventionalTransverse   EN   Conventional FK processing for transverse projections.\n"
         "#  ConventionalRayleigh     ENZ  Conventional FK processing for radial projections.\n"
         "#  Capon                    Z    High resolution FK processing (Capon, 1969).\n"
         "#  CaponRadial              EN   High resolution FK processing (Capon, 1969) for radial projections.\n"
         "#  CaponTransverse          EN   High resolution FK processing (Capon, 1969) for transverse projections.\n"
         "#  RTBF                     ENZ  According to Wathelet et al (2018).\n"
         "#                                Cross spectrum made of radial projections and vertical.\n"
         "#                                Combined radial and vertical ellipticity steering.\n"
         "#  RTBFRadial               ENZ  According to Wathelet et al (2018).\n"
         "#                                Cross spectrum made of radial projections and vertical.\n"
         "#                                Radial ellipticity steering, better for small ellipticities.\n"
         "#  PoggiVertical            ENZ  According Poggi et al. (2010)\n"
         "#                                k picked from vertical processing.\n"
         "#  PoggiRadial              ENZ  According Poggi et al. (2010)\n"
         "#                                k picked from radial processing.\n"
         "#  RDS                      ENZ  Rayleigh Direct Steering.\n"
         "#                                Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial projections included in the steering matrix.\n"
         "#                                Combined radial and vertical ellipticity steering.\n"
         "#  LDS                      EN   Love Direct Steering.\n"
         "#                                Cross spectrum made of raw components E and N.\n"
         "#                                Transverse projections included in the steering matrix.\n"
         "#  RDSRadial                ENZ  Rayleigh Direct Steering.\n"
         "#                                Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial projections included in the steering matrix.\n"
         "#                                Radial ellipticity steering, better for small ellipticities.\n"
         "#  Experimental modes:\n"
         "#  RTBFFixedEll             ENZ  Same as RTBF but ellipticity is fixed.\n"
         "#                                FIXED_ELLIPTICITY_FILE_NAME must be provided.\n"
         "#  RDSFixedEll              ENZ  Same as RDS but ellipticity is fixed.\n"
         "#                                FIXED_ELLIPTICITY_FILE_NAME must be provided.\n"
         "#  ActiveConventional       Z    Conventional FK processing for active sources.\n"
         "#  ActiveRTBF               ENZ  RTBF for active sources\n"
         "#                                Cross spectrum made of radial and transverse projections.\n"
         "#  ActiveDS                 ENZ  Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial and transverse projections included in steering matrix.\n"
         "#  ActiveConventional            Conventional FK processing for active sources\n"
         "#                                Cross spectrum made of radial and transverse projections.\n"
         "#  Omni                     ENZ  Same cross spectrum as RDS.\n"
         "#                                Ouput power is the sum of power in all directions\n"
         "#  RDSVertical              ENZ  Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial and transverse projections included in steering matrix.\n"
         "#                                Radial ellipticity steering.\n"
         "#  RDSRadial                ENZ  Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial and transverse projections included in steering matrix.\n"
         "#                                Vertical ellipticity steering.\n"
         "#  RDSRefined               ENZ  Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial and transverse projections included in steering matrix.\n"
         "#                                Iterative ellitpticity assessment.\n";
    log+=prefix+"PROCESS_TYPE"+suffix+"="+processTypeString()+"\n";
    switch(_processType) {
    case RTBF:
    case RTBFRadial:
    case RTBFFixedEll:
    case PoggiVertical:
    case PoggiRadial:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
    case CaponRadial:
    case CaponTransverse:
      log+="# Number of steps for the computation of radial/transverse projections\n";
      log+=prefix+"ROTATE_STEP_COUNT"+suffix+"="+QString::number(_rotateStepCount)+"\n";
      break;
    default:
      break;
    }
    log+=prefix+"DAMPING_FACTOR"+suffix+"="+QString::number(_damping)+"\n";
    log+="# If provided and PROCESS_TYPE==DirectSteering, the ellipticity is forced to the provided curve.\n"
         "# The file must contain two columns: frequency and signed ellipticity.\n"
         "# Provided sampling must not necessarily match the processing sampling frequency, linear interpolation is used.\n"
         "# Better for precision if the two sampling match.\n"
         "# To generate a synthetic curve: gpell M2.1.model -one-mode -R 1 -min 0.5 -max 50 -n 187 > curve.txt\n";
    log+=prefix+"FIXED_ELLIPTICITY_FILE_NAME"+suffix+"="+_fixedEllipticityFileName+"\n";
    log+="# Minimum distance between source and receiver (for active source only)\n";
    log+=prefix+"MINIMUM_DISTANCE"+suffix+"="+QString::number(_minimumDistance)+"\n";
    log+="# Maximum distance between source and receiver (for active source only)\n";
    log+=prefix+"MAXIMUM_DISTANCE"+suffix+"="+QString::number(_maximumDistance)+"\n";
    log+=prefix+"SOURCE_GRID_STEP"+suffix+"="+QString::number(_sourceGridStep)+"\n";
    log+=prefix+"SOURCE_GRID_SIZE"+suffix+"="+QString::number(_sourceGridSize)+"\n";
    log+="# Experimental join processing of several arrays\n";
    log+="# Several ARRAY can be defined with a list of station names\n";
    for(QList<QStringList>::const_iterator it=_arrays.begin(); it!=_arrays.end(); it++) {
      log+=prefix+"ARRAY"+suffix+"="+it->join(',')+"\n";
    }
    log+="#\n"
         "#\n"
         "#     Wavenumber grid\n"
         "#\n"
         "#\n";
    log+="# Wavenumber fine gridding used as a cache for the FK maps\n";
    log+=prefix+"CACHE_GRID_STEP"+suffix+" (rad/m)="+QString::number(_cacheGridStep)+"\n";
    log+="# If CACHE_GRID_STEP is null, GRID_STEP is computed from K_MIN*CACHE_GRID_STEP_FACTOR.\n";
    log+=prefix+"CACHE_GRID_STEP_FACTOR"+suffix+"="+QString::number(_cacheGridStepFactor)+"\n";

    log+="# Wavenumber coarse gridding used for searching maxima of the FK maps\n";
    log+=prefix+"GRID_STEP"+suffix+" (rad/m)="+QString::number(_gridStep)+"\n";
    log+="# If GRID_STEP is null, GRID_STEP is computed from K_MIN*GRID_STEP_FACTOR.\n";
    log+=prefix+"GRID_STEP_FACTOR"+suffix+"="+QString::number(_gridStepFactor)+"\n";
    log+=prefix+"GRID_SIZE"+suffix+" (rad/m)="+QString::number(_gridSize)+"\n";
    log+="# If GRID_SIZE is null, GRID_SIZE is computed from K_MAX*GRID_SIZE_FACTOR.\n";
    log+=prefix+"GRID_SIZE_FACTOR"+suffix+"="+QString::number(_gridSizeFactor)+"\n";
    log+="# Effective GRID_STEP is "+QString::number(gridStep())+".\n";
    log+="# Effective GRID_SIZE is "+QString::number(gridSize())+".\n";

    log+="# Minimum velocity of the searched maxima of the FK map\n";
    log+=prefix+"MIN_V"+suffix+" (m/s)="+QString::number(1.0/_smax)+"\n";
    log+="# Maximum velocity of the searched maxima of the FK map\n";
    log+=prefix+"MAX_V"+suffix+" (m/s)="+QString::number(1.0/_smin)+"\n";

    log+="# Minimum azimuth of the searched maxima of the FK map (clockwise from North)\n";
    log+=prefix+"MIN_AZIMUTH"+suffix+" (deg.)="+QString::number(_minAzimuth)+"\n";
    log+="# Maximum azimith of the searched maxima of the FK map (clockwise from North)\n";
    log+=prefix+"MAX_AZIMUTH"+suffix+" (deg.)="+QString::number(_maxAzimuth)+"\n";

    log+="# Theoretical Kmin and Kmax computed from array geometry\n";
    log+="# Used only for post-processing\n";
    log+=prefix+"K_MIN"+suffix+" (rad/m)="+QString::number(_kmin)+"\n";
    log+=prefix+"K_MAX"+suffix+" (rad/m)="+QString::number(_kmax)+"\n";
    log+="#\n"
         "#\n"
         "#     Peak picking\n"
         "#\n"
         "#\n";
    log+="# Inversion method used for getting FK peaks: Gradient or RefinedGrid\n";
    log+=prefix+"INVERSION_METHOD"+suffix+"="+inversionMethodString()+"\n";
    log+=prefix+"N_MAXIMA"+suffix+"="+QString::number(_maxPeakCount)+"\n";
    log+=prefix+"ABSOLUTE_THRESHOLD"+suffix+"="+QString::number(_absoluteThreshold)+"\n";
    log+=prefix+"RELATIVE_THRESHOLD"+suffix+" (%)="+QString::number(_relativeThreshold)+"\n";
    log+=prefix+"SAVE_UNDEFINED_ELLIPTICITIES"+suffix+"="+(_saveUndefinedEllipticities ? "y" : "n")+"\n";
    log+=prefix+"EXPORT_ALL_FK_GRIDS"+suffix+"="+(_exportAllFKGrids ? "y" : "n")+"\n";
    return log;
  }

  bool FKParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    bool ok=true;
    switch(index-ArrayParameters::keywordCount()) {
    case 19:
      setInversionMethod(value);
      return true;
    case 20:
      _cacheGridStep=value.toDouble(&ok);
      return ok;
    case 21:
      _cacheGridStepFactor=value.toDouble(&ok);
      return ok;
    case 0:
      if(version()<1) {
        _gridStep=value.toDouble(&ok); // Old release were saving grid_step instead of kmin
        _kmin=0.0;                     // Force automatic computation from array geometry
      } else {
        _kmin=value.toDouble(&ok);
      }
      return ok;
    case 1:
      if(version()<1) {
        _gridSize=value.toDouble(&ok); // Old release were saving grid_size instead of kmax
        _kmax=0.0;                     // Force automatic computation from array geometry
      } else {
        _kmax=value.toDouble(&ok);
      }
      return ok;
    case 16:
      _gridStep=value.toDouble(&ok);
      return ok;
    case 17:
      _gridSize=value.toDouble(&ok);
      return ok;
    case 18:
      _gridStepFactor=value.toDouble(&ok);
      return ok;
    case 28:
      _gridSizeFactor=value.toDouble(&ok);
      return ok;
    case 24:
      _smin=1.0/value.toDouble(&ok);
      return ok;
    case 2:
      _smax=1.0/value.toDouble(&ok);
      return ok;
    case 25:
      _minAzimuth=value.toDouble(&ok);
      return ok;
    case 26:
      _maxAzimuth=value.toDouble(&ok);
      return ok;
    case 3:
      _maxPeakCount=value.toInt(&ok);
      return ok;
    case 4:
      _absoluteThreshold=value.toDouble(&ok);
      return ok;
    case 5:
      _relativeThreshold=value.toDouble(&ok);
      return ok;
    case 6:
      _exportAllFKGrids=(value=="y");
      return true;
    case 27:
      _saveUndefinedEllipticities=(value=="y");
      return ok;
    case 7:
      if(_damping>=0.0) {
        _damping=value.toDouble(&ok);
      }
      return ok;
    case 8:
      _rotateStepCount=value.toInt(&ok);
      return ok;
    case 9:
      setProcessType(value);
      return true;
    case 10:
      _arrays.append(value.split(','));
      return true;
    case 11:  // For compatibility
      obsoleteKeyword(keywords, 11);
      return true;
    case 12:
      _fixedEllipticityFileName=value;
      return true;
    case 13:
      _minimumDistance=value.toDouble(&ok);
      return ok;
    case 14:
      _maximumDistance=value.toDouble(&ok);
      return ok;
    case 15: // For compatibility
      obsoleteKeyword(keywords, 15);
      if(value=="n") {
        _damping=-1.0;
      }
      return true;
    case 22:
      _sourceGridStep=value.toDouble(&ok);
      return ok;
    case 23:
      _sourceGridSize=value.toDouble(&ok);
      return ok;
    default:
      break;
    }
    return ArrayParameters::setValue(index, value, unit, keywords);
  }

  bool FKParameters::setProcessType(QString p)
  {
    p=p.toLower();
    if(p.count()<3) {
      return false;
    }
    switch(p[2].unicode()) {
    case 'b':
      if(p=="rtbf") {
        _processType=RTBF;
      } else if(p=="rtbfradial") {
        _processType=RTBFRadial;
      } else if(p=="rtbffixedell") {
        _processType=RTBFFixedEll;
      }
      return false;
    case 'g':
      if(p=="poggivertical") {
        _processType=PoggiVertical;
      } else if(p=="poggiradial") {
        _processType=PoggiRadial;
      }
      return false;
    case 'n':
      if(p=="conventional") {
        _processType=Conventional;
      } else if(p=="omni") {
        _processType=Omni;
      } else if(p=="conventionalradial") {
        _processType=ConventionalRadial;
      } else if(p=="conventionaltransverse") {
        _processType=ConventionalTransverse;
      } else if(p=="conventionalrayleigh") {
        _processType=ConventionalRayleigh;
      }
      return false;
    case 'o':
      if(p=="projections") {                     // Kept for compatibility
        _processType=RTBF;
      }
      return false;
    case 'p':
      if(p=="capon") {
        _processType=Capon;
      } else if(p=="caponradial") {
        _processType=CaponRadial;
      } else if(p=="capontransverse") {
        _processType=CaponTransverse;
      }
      return false;
    case 'r':
      if(p=="directsteering") {                  // Kept for compatibility
        _processType=RDS;
      } else if(p=="directsteeringrefined") {    // Kept for compatibility
        _processType=RDSRefined;
      } else if(p=="directsteeringvertical") {   // Kept for compatibility
        _processType=RDSVertical;
      } else if(p=="directsteeringradial") {    // Kept for compatibility
        _processType=RDSRadial;
      } else if(p=="directsteeringsingle") {    // Kept for compatibility
        _processType=RDSSingle;
      }
      return false;
    case 's':
      if(p=="rds") {
        _processType=RDS;
      } else if(p=="lds") {
        _processType=LDS;
      } else if(p=="rdsradial") {
        _processType=RDSRadial;
      } else if(p=="rdsvertical") {
        _processType=RDSVertical;
      } else if(p=="rdsrefined") {
        _processType=RDSRefined;
      } else if(p=="rdssingle") {
        _processType=RDSSingle;
      } else if(p=="rdsfixedell") {
        _processType=RDSFixedEll;
      }
      return false;
    case 't':
      if(p=="activeconventional") {
        _processType=ActiveConventional;
      } else if(p=="activecapon") {
        _processType=ActiveCapon;
      } else if(p=="activeconventionalrayleigh") {
        _processType=ActiveConventionalRayleigh;
      } else if(p=="activertbf") {
        _processType=ActiveRTBF;
      } else if(p=="activeds") {
        _processType=ActiveDS;
      } else if(p=="activeprojections") {        // Kept for compatibility
        _processType=ActiveRTBF;
      } else if(p=="activedirectsteering") {     // Kept for compatibility
        _processType=ActiveDS;
      }
      return false;
    }
    return true;
  }

  QString FKParameters::processTypeString() const
  {
    switch(_processType) {
    case Conventional:
      break;
    case Capon:
      return "Capon";
    case ActiveConventional:
      return "ActiveConventional";
    case ActiveCapon:
      return "ActiveCapon";
    case Omni:
      return "Omni";
    case ConventionalRadial:
      return "ConventionalRadial";
    case CaponRadial:
      return "CaponRadial";
    case ConventionalTransverse:
      return "ConventionalTransverse";
    case CaponTransverse:
      return "CaponTransverse";
    case ConventionalRayleigh:
      return "ConventionalRayleigh";
    case RTBF:
      return "RTBF";
    case RTBFRadial:
      return "RTBFRadial";
    case RTBFFixedEll:
      return "RTBFFixedEll";
    case RDS:
      return "RDS";
    case RDSRadial:
      return "RDSRadial";
    case RDSVertical:
      return "RDSVertical";
    case RDSRefined:
      return "RDSRefined";
    case RDSSingle:
      return "RDSSingle";
    case RDSFixedEll:
      return "RDSFixedEll";
    case PoggiVertical:
      return "PoggiVertical";
    case PoggiRadial:
      return "PoggiRadial";
    case LDS:
      return "LDS";
    case ActiveConventionalRayleigh:
      return "ActiveConventionalRayleigh";
    case ActiveRTBF:
      return "ActiveRTBF";
    case ActiveDS:
      return "ActiveDS";
    }
    return "Conventional";
  }

  bool FKParameters::isActiveProcessType() const
  {
    switch(_processType) {
    FKPARAMETERS_PASSIVE_PROCESS_TYPES
      return false;
    FKPARAMETERS_ACTIVE_PROCESS_TYPES
      break;
    }
    return true;
  }


  bool FKParameters::setInversionMethod(QString p)
  {
    p=p.toLower();
    if(p=="refinedgrid") {
      _inversionMethod=RefinedGrid;
    } else if(p=="gradient") {
      _inversionMethod=Gradient;
    } else {
      return false;
    }
    return true;
  }

  QString FKParameters::inversionMethodString() const
  {
    switch(_inversionMethod) {
    case Gradient:
      break;
    case RefinedGrid:
      return "RefinedGrid";
    }
    return "Gradient";
  }

  QString FKParameters::outputName(const QString& groupName) const
  {
    QString fn;
    if(ArrayParameters::outputBaseName().endsWith(".max")) {
      return ArrayParameters::outputBaseName();
    } else if(ArrayParameters::outputBaseName().isEmpty()) {
      return processTypeString().toLower()+"-"+groupName;
    } else {
      QString baseName;
      if(!ArrayParameters::outputBaseName().endsWith("-") &&
         !ArrayParameters::outputBaseName().endsWith("_")) {
        baseName=ArrayParameters::outputBaseName()+"-";
      } else {
        baseName=ArrayParameters::outputBaseName();
      }
      if(baseName.contains(processTypeString().toLower())) {
        return baseName+groupName;
      } else {
        return baseName+processTypeString().toLower()+"-"+groupName;
      }
    }
  }

} // namespace ArrayCore
