/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-12-02
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef FKCROSSSPECTRUM_H
#define FKCROSSSPECTRUM_H

#include "ArrayCoreDLLExport.h"
#include "ArrayCrossSpectrum.h"
#include "AbstractFKFunction.h"
#include "ActiveFKSteering.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT FKCrossSpectrum : public ArrayCrossSpectrum
  {
  public:
    FKCrossSpectrum(const ArraySelection * array,
                    const FKParameters * param);
    ~FKCrossSpectrum();

    const FKParameters * parameters() const {return static_cast<const FKParameters *>(_parameters);}

    double rotationStep() const {return _rotateStep;}
    int rotationStepCount() const {return _rotateStepCount;}

    bool calculate(const QVector<int>& blocks, AbstractFKFunction * function);

    void setActiveSource(const SeismicEvent * src) {_source=src;}
    Point setActiveSource(const QVector<int>& blocks, bool& ok);

    const Point& sourceOffset() const {return _sourceOffset;}
    void setSourceOffset(const Point& o) {_sourceOffset=o;}

    const ArraySelection& array() const {return _array;}

    void resetRotatedMatrices();

    void addThreeComponent();
    // Test of optical fiber arrays
    //void addHorizontalRotatedRayleigh() {addHorizontalRotated(_rayleigh);}
    //void addHorizontalRotatedLove() {addHorizontalRotated(_love);}
    void addRotatedRadial();
    void addRotatedTransverse();
    void addRotatedRayleigh();
    void addActiveVertical(const ActiveFKSteering& geom);
    void addActiveRayleigh(const ActiveFKSteering& geom);
    void addActiveTransverse(const ActiveFKSteering& geom);
    void addActiveThreeComponent(const ActiveFKSteering& geom);
    void addRayleighSingle();

    void meanRotatedMatrices(int nBlocks);
    void meanRayleighSingle();

    bool invert();
    bool invertRotatedMatrices();

    inline const ComplexMatrix& rotatedMatrix(const Point2D& k) const;
    inline const ComplexMatrix& rotatedMatrix(int rotationIndex) const;

    bool setMask();
    void setEllipticity(const ComplexMatrix& ell);
  private:
    void resize();
    // Test of optical fiber arrays
    //void addHorizontalRotated(ComplexMatrix& covmat);
    void addRotatedRadial(int rotationIndex);
    void addRotatedTransverse(int rotationIndex);
    void addRotatedRayleigh(int rotationIndex);
    inline int rotationIndex(const Point2D& k) const;
    void normalize(ComplexMatrix& covmat);
    bool invert(ComplexMatrix& covmat);

    bool setRayleighArrayMask();
    bool setOneComponentArrayMask();

    int _rotateStepCount;
    double _rotateStep, _invRotateStep;

    ArraySelection _array;
    const SeismicEvent * _source;
    Point _sourceOffset;

    DoubleMatrix _arrayMask;
    ComplexMatrix * _rotatedMatrices;
  };

  inline int FKCrossSpectrum::rotationIndex(const Point2D& k) const
  {
    double angle=::atan2(k.y(), k.x()); // From -M_PI to M_PI
    if(angle<0.0) {
      angle+=2.0*M_PI;
    }
    // Angle from 0 to M_PI
    int index=qRound(angle*_invRotateStep);
    if(index<_rotateStepCount) {
      return index;
    } else {
      return 0;
    }
  }

  inline const ComplexMatrix& FKCrossSpectrum::rotatedMatrix(const Point2D& k) const
  {
    return _rotatedMatrices[rotationIndex(k)];
  }

  inline const ComplexMatrix& FKCrossSpectrum::rotatedMatrix(int rotationIndex) const
  {
    return _rotatedMatrices[rotationIndex];
  }

} // namespace ArrayCore

#endif // FKCROSSSPECTRUM_H

